package ru.yandex.autotests.direct.web.steps.transfer;

import org.hamcrest.Matcher;

import ru.yandex.autotests.direct.utils.beans.IBeanWrapper;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.web.objects.transfer.TransferMoney;
import ru.yandex.autotests.direct.web.objects.transfer.TransferType;
import ru.yandex.autotests.direct.web.pages.transfer.TransferPage;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.webdriver.pages.BasePageObject;
import ru.yandex.qatools.allure.webdriver.steps.BaseSteps;
import ru.yandex.qatools.allure.webdriver.steps.WebDriverSteps;

import static org.hamcrest.CoreMatchers.anyOf;
import static org.hamcrest.CoreMatchers.containsString;
import static ru.yandex.autotests.direct.web.util.matchers.webdriver.HasURL.hasURL;


/**
 * User: buhter
 * Date: 11.12.12
 * Time: 18:14
 * Общие степы переноса средств
 */
public class TransferSteps extends BaseSteps {

    private TransferPage onTransferPage() {
        return BasePageObject.getInstance(TransferPage.class, config);
    }


    /**
     * Выбор типа переноса.
     *
     * @param transferType - тип переноса
     */
    @Step("Выбираем тип перноса средств - {0}")
    public void selectTransferType(TransferType transferType) {
        onTransferPage().chooseTransferType(transferType);
    }

    /**
     * Проверка серверной ошибки при попытке переноса средств
     *
     * @param matcher - матчер на ошибку
     */
    @Step("Проверяем ошибку переноса средств")
    public void shouldSeeTransferError(Matcher matcher) {
        onTransferPage().checkTransferError(matcher);
    }

    /**
     * Проверка JS алерта при попытке переноса средств
     *
     * @param matcher - matcher текста ожидаемого алерта
     */
    @Step("Проверяем предупреждение при некорректном переносе средств")
    public void shouldSeeAlertWithErrorAfterSubmit(Matcher<String> matcher) {
        onTransferPage().onTransferForm().submitTransferForm();
        onTransferPage().acceptAlert(matcher);
    }

    /**
     * Степ заполнения и проверки формы переноса средств
     *
     * @param transferMoney - бин для заполнения формы
     * @param moneyCurrency - MoneyCurrency клиента, нужна чтобы проверить форму
     *                      в соответствии с локалью и валютой клиента
     */
    @Step("Вносим в форму переноса средств {0} и проверяем корректность данных {0} и {1}")
    public void fillTransferForm(IBeanWrapper<TransferMoney> transferMoney, MoneyCurrency moneyCurrency) {
        onTransferPage().onTransferForm().fillTransferForm(transferMoney.getBean());
        onTransferPage().onTransferForm().checkTransferForm(transferMoney.getBean(), moneyCurrency);
    }

    /**
     * Степ проверки формы переноса средств.
     *
     * @param transferMoney - бин для проверки формы
     * @param moneyCurrency - MoneyCurrency клиента, нужна чтобы проверить форму
     *                      в соответствии с локалью и валютой клиента
     */
    @Step("Проверяем форму переноса средств, данные {0} и {1}")
    public void shouldSeeInTransferForm(IBeanWrapper<TransferMoney> transferMoney, MoneyCurrency moneyCurrency) {
        onTransferPage().onTransferForm().checkTransferForm(transferMoney.getBean(), moneyCurrency);
    }

    @Step("Проверяем, что доступные для переноса средства для кампании {0} удовлетворяют условию {1}")
    public void shouldSeeAvailableTransferMoney(Long campaignId, Matcher<String> matcher) {
        onTransferPage().onTransferForm().checkAvailableTransferMoney(campaignId, matcher);
    }

    @Step("Получаем сумму, доступную для переноса, для кампании {0}")
    public String getAvailableTransferMoney(Long campaignId) {
        return onTransferPage().onTransferForm().getAvailableTransferMoney(campaignId);
    }

    @Step("Проверяем, что общая сумма для переноса соответствует {0}")
    public void shouldSeeTotalTransferValue(Matcher matcher) {
        onTransferPage().onTransferForm().shouldSeeTotalTransferValue(matcher);
    }

    /**
     * Степ заполнения БЕЗ проверки формы переноса средств
     *
     * @param transferMoney - бин для заполнения формы
     */
    @Step("Вносим в форму переноса средств {0} не проверяя")
    public void fillTransferFormWithoutVerify(IBeanWrapper<TransferMoney> transferMoney) {
        onTransferPage().onTransferForm().fillTransferForm(transferMoney.getBean());
    }

    /**
     * Степ трансфера. Нажимает на кнопку переноса средств и дожидается окончания процедуры
     */
    @Step("Отправляем форму переноса средств")
    public void performTransfer() {
        onTransferPage().onTransferForm().submitTransferForm();
        //Чтобы заказ успел дойти до баланса перед синхронизацией.
        try {
            Thread.sleep(1000);
        } catch (InterruptedException e) {
            throw new RuntimeException("Не смог подождать", e);
        }
    }

    @Step("Выбираем перенос со всех доступных кампаний")
    public void clickTransferFromAllCampaignsOption() {
        onTransferPage().onTransferForm().clickTransferFromAllCampaignsOption();
    }

    @Step("Распределяем срдества на все доступные кампании")
    public void clickDistributeToCampaignsOption() {
        onTransferPage().onTransferForm().clickDistributeToCampaignsOption();
    }

    @Step("Нажимаем кнопку переноса средств")
    public void clickTransferButton() {
        onTransferPage().onTransferForm().submitTransferForm();
    }

    @Step("Выбираем перенос на все доступные кампании")
    public void clickSelectAllToCampaignsOption() {
        onTransferPage().onTransferForm().clickSelectAllToCampaignsOption();
    }

    @Step("Очишаем таблицу переноса средств 'откуда'")
    public void clickClearFromCampaignsOption() {
        onTransferPage().onTransferForm().clickClearFromCampaignsOption();
    }

    @Step("Очищаем таблицу переноса средств 'куда'")
    public void clickClearToCampaignsOption() {
        onTransferPage().onTransferForm().clickClearToCampaignsOption();
    }

    @Step("Проверяем, что таблица переноса средств 'откуда' пустая")
    public void shouldSeeEmptyFromTable(MoneyCurrency moneyCurrency) {
        onTransferPage().onTransferForm().shouldSeeEmptyFromTable(moneyCurrency);
    }

    @Step("Проверяем, что таблица переноса средств 'куда' пустая")
    public void shouldSeeEmptyToTable(MoneyCurrency moneyCurrency) {
        onTransferPage().onTransferForm().shouldSeeEmptyToTable(moneyCurrency);
    }

    @Step("Ждем завершение переноса")
    public void waitForSubmittingFinish() {
        getInstance(WebDriverSteps.class, config).waitForPage(hasURL(anyOf(containsString("cmd=showCamps"),
                containsString("cmd=showClients"))), 310);
    }
}