package ru.yandex.autotests.direct.web.util.beanutils;

import java.beans.PropertyDescriptor;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.ConversionException;
import org.apache.commons.beanutils.PropertyUtils;

import ru.yandex.autotests.direct.web.util.DirectWebError;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.webdriver.utils.WebDriverLog;

import static java.beans.Beans.instantiate;
import static org.apache.commons.beanutils.ConvertUtils.convert;
import static org.apache.commons.beanutils.PropertyUtils.getPropertyType;

/**
 * @author Roman Kuhta (kuhtich@yandex-team.ru)
 */
public class BeanHelper {
    /**
     * Checks if object has nut null properties, this$0 is ignored
     *
     * @param object Any object
     * @param <T>    Object type
     * @return Returns true if one of declared fields has not null value
     */
    public static <T> boolean hasNotNullProperty(T object) {
        List<PropertyDescriptor> propertyDescriptors = Stream.of(PropertyUtils.getPropertyDescriptors(object))
                .filter(x -> x.getReadMethod() != null)
                .collect(Collectors.toList());
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            String childName = propertyDescriptor.getName();
            if (childName.startsWith("__") || childName.equals("class")) {
                continue;
            }
            try {
                if (propertyDescriptor.getReadMethod().invoke(object) != null) {
                    return true;
                }
            } catch (IllegalAccessException e) {
                throw new DirectWebError("Не удалось получить доступ к полю " + childName);
            } catch (InvocationTargetException e) {
                LogSteps.getLogger(WebDriverLog.class).info("Exception: " + e.getMessage());
            }
        }
        return false;
    }

    public static <T> T getOnlyFields(T bean, String... fields) {
        //noinspection unchecked
        T result = null;
        try {
            result = (T) instantiate(BeanHelper.class.getClassLoader(), bean.getClass().getName());
        } catch (IOException | ClassNotFoundException e) {
            throw new DirectWebError(e);
        }
        for (String fieldName : fields) {
            try {
                Object value = getProperty(bean, fieldName);
                setNotNullProperty(result, fieldName, value);
            } catch (Exception e) {
                throw new DirectWebError(e);
            }
        }
        return result;
    }

    public static <T> T getOnlySimpleFields(T bean) {
        return getOnlyFields(bean, getOnlySimpleFieldsName(bean));
    }

    public static boolean isSimpleField(Class clazz) {
        return clazz.equals(String.class) || clazz.isEnum() ||
                clazz.isPrimitive() || Number.class.isAssignableFrom(clazz);
    }

    public static <T> String[] getOnlySimpleFieldsName(T bean) {
        List<String> fields = new ArrayList<>();
        PropertyDescriptor[] descriptors = PropertyUtils.getPropertyDescriptors(bean);
        for (PropertyDescriptor propertyDescriptor : descriptors) {
            if (propertyDescriptor.getReadMethod() != null &&
                    propertyDescriptor.getWriteMethod() != null &&
                    isSimpleField(propertyDescriptor.getPropertyType())) {
                fields.add(propertyDescriptor.getName());
            }
        }
        return fields.toArray(new String[fields.size()]);
    }

    public static <T> void setNotNullProperty(T bean, String fieldName, Object value) {
        if (value == null)
            return;
        try {
            BeanUtils.setProperty(bean, fieldName, convert(value, getPropertyType(bean, fieldName)));
        } catch (ConversionException e) {
        } catch (Exception e) {
            throw new DirectWebError(e);
        }
    }

    /**
     * Метод установит поле бина в том случае, если в ожидаемом бине это свойство не null
     *
     * @param bean         Объект, в котором устанавливается поле
     * @param expectedBean Ожидаемый объект
     * @param fieldName    название поля объекта
     * @param value        значение поля
     * @param <T>          класс объекта
     */
    public static <T> void setNotNullProperty(T bean, T expectedBean, String fieldName, Object value) {
        if (getProperty(expectedBean, fieldName) == null)
            return;
        setNotNullProperty(bean, fieldName, value);
    }

    public static void copyProperties(Object dest, Object orig) {
        try {
            PropertyUtils.copyProperties(dest, orig);
        } catch (Exception e) {
            throw new DirectWebError(e);
        }
    }

    public static Object getProperty(Object bean, String name) {
        Method method;
        try {
            method = new PropertyDescriptor(name, bean.getClass()).getReadMethod();
            return method.invoke(bean);
        } catch (Exception e) {
            throw new DirectWebError(e);
        }
    }

    public static Boolean hasSetOneOfProperties(Object bean, String... propertyNames) {
        for (String propertyName : propertyNames) {
            if (getProperty(bean, propertyName) != null)
                return true;
        }
        return false;
    }
}
