package ru.yandex.autotests.direct.web.util.matchers;

import java.math.BigDecimal;
import java.math.RoundingMode;

import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.TypeSafeMatcher;

/**
 * User: kuhtich, xy6er
 * Date: 12.09.13
 */
public class NumberApproximatelyEqual<T extends Number> extends TypeSafeMatcher<T> {
    private BigDecimal excepted;
    private BigDecimal diff;
    private double percentageDiff;

    private NumberApproximatelyEqual(T excepted) {
        this.excepted = new BigDecimal(excepted.toString());
    }

    public NumberApproximatelyEqual<T> withDifference(T difference) {
        this.diff = new BigDecimal(difference.toString()).setScale(excepted.scale(), RoundingMode.HALF_UP);
        this.percentageDiff = diff.doubleValue() * 100 / excepted.doubleValue();
        return this;
    }

    public NumberApproximatelyEqual<T> withDifferenceInPercents(double percentageDiff) {
        this.percentageDiff = percentageDiff;
        this.diff = excepted.multiply(new BigDecimal(percentageDiff / 100)).
                setScale(excepted.scale(), RoundingMode.HALF_UP);
        return this;
    }

    private String formatPercent(double percentage) {
        return "(" + String.format("%.2f", percentage) + "%)";
    }

    private String formatBigDecimal(BigDecimal bigDecimal) {
        return "<" + bigDecimal.stripTrailingZeros().toPlainString() + ">";
    }

    @Override
    protected boolean matchesSafely(T actual) {
        // Math.abs( expected - actual ) <= diff;
        return excepted.subtract(new BigDecimal(actual.toString())).abs().compareTo(diff) != 1;
    }

    @Override
    public void describeTo(Description description) {
        description.appendText("a value approximately equal to ").
                appendText(formatBigDecimal(excepted)).
                appendText(" with difference not exceeds ").
                appendText(formatBigDecimal(diff)).
                appendText(formatPercent(percentageDiff));
    }

    @Override
    protected void describeMismatchSafely(T item, Description mismatchDescription) {
        BigDecimal actual = new BigDecimal(item.toString());
        BigDecimal actualDiff = excepted.subtract(actual).abs().setScale(excepted.scale(), RoundingMode.HALF_UP);
        double actualPercentageDiff = actualDiff.doubleValue() * 100 / excepted.doubleValue();
        mismatchDescription.appendText("the difference between expected").
                appendText(formatBigDecimal(excepted)).
                appendText(" and actual").
                appendText(formatBigDecimal(actual)).
                appendText(" was ").
                appendText(formatBigDecimal(actualDiff)).
                appendText(formatPercent(actualPercentageDiff));
    }

    @Factory
    public static <T extends Number> NumberApproximatelyEqual<T> approxEqualTo(T excepted) {
        return new NumberApproximatelyEqual<>(excepted);
    }

}