package ru.yandex.autotests.direct.web.util.rules;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.junit.internal.AssumptionViolatedException;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.model.MultipleFailureException;
import org.junit.runners.model.Statement;
import org.openqa.grid.common.exception.GridException;
import org.openqa.selenium.InvalidElementStateException;
import org.openqa.selenium.StaleElementReferenceException;
import org.openqa.selenium.UnhandledAlertException;
import org.openqa.selenium.WebDriverException;
import org.openqa.selenium.remote.ErrorHandler;
import org.openqa.selenium.remote.SessionNotFoundException;

import ru.yandex.autotests.direct.web.util.DirectWebException;
import ru.yandex.autotests.direct.web.util.ElementNotVisibleException;
import ru.yandex.autotests.direct.web.util.NoSuchElementException;

import static java.util.Objects.isNull;
import static org.apache.commons.lang3.exception.ExceptionUtils.getRootCause;

/**
 * Created with IntelliJ IDEA.
 * User: alexey-n
 */
public class ExceptionWrapperRule implements TestRule {

    @Override
    public Statement apply(final Statement base, final Description description) {
        return new Statement() {
            @Override
            public void evaluate() throws Throwable {
                try {
                    base.evaluate();
                } catch (AssertionError | AssumptionViolatedException | MultipleFailureException e) {
                    throw e;
                } catch (Throwable t) {
                    throw getExceptionSafely(t);
                }
            }
        };
    }

    private String getElementSelector(Throwable t) {
        if (t.getMessage() != null && !"".equals(t.getMessage())) {
            return t.getMessage().split("\n")[0];
        }
        return "";
    }

    private Throwable getExceptionSafely(Throwable t) {
        try {
            return getException(t);
        } catch (Throwable e) {
            return t;
        }
    }

    private Throwable getException(Throwable t) {
        Throwable rootCause = getRootCause(t);
        rootCause = isNull(rootCause) ? t : rootCause;

        if (rootCause instanceof org.openqa.selenium.NoSuchElementException) {
            String selectorOfElement;
            try {
                selectorOfElement = getElementSelector(rootCause);
            } catch (Exception e) {
                selectorOfElement = "Can't get element selector.";
            }
            return new NoSuchElementException(selectorOfElement, t);
        }

        if (rootCause instanceof org.openqa.selenium.ElementNotVisibleException) {
            return new ElementNotVisibleException("Element is present in html but not visible", rootCause);
        }

        if (rootCause instanceof UnhandledAlertException) {
            return new DirectWebException("Unexpected alert with message: " +
                    ((UnhandledAlertException) rootCause).getAlertText(), rootCause);
        }

        if (t instanceof WebDriverException || rootCause instanceof WebDriverException) {
            if (rootCause instanceof StaleElementReferenceException) {
                return new DirectWebException("Problem with page element. Looks like bug in tests", t);
            }
            if (rootCause instanceof InvalidElementStateException) {
                return new DirectWebException("Problem with page element state(visibility). Looks like bug in tests",
                        t);
            }
            if (rootCause instanceof UnhandledAlertException) {
                return new DirectWebException("Some unexpected alert appeared with text: " +
                        ((UnhandledAlertException) t).getAlertText(), t);
            }
            if (rootCause instanceof ErrorHandler.UnknownServerException) {
                return new DirectWebException("Some exception with selenium grid server", t);
            }
            if (rootCause instanceof SessionNotFoundException) {
                return new DirectWebException("Selenium grid session not found. "
                        + "Seems like error appeared in the previous step", t);
            }
            if (t.getMessage().contains("Command duration")) {
                Pattern p = Pattern.compile("Command duration.*");
                Matcher m = p.matcher(t.getMessage());
                if (m.find()) {
                    return new DirectWebException("Grid session timeout: " + m.group(), t);
                }
            }
            if (rootCause.getMessage().contains("Timed out when searching for valid host") ||
                    rootCause instanceof GridException)
            {
                if (rootCause.getMessage().contains("was terminated due to TIMEOUT")) {
                    return new DirectWebException("Application response is too long", t);
                }
                return new DirectWebException("Selenium grid server error", t);
            }
            return new DirectWebException("Some web driver exception. See stacktrace for detail", t);
        }

        if (t instanceof NoClassDefFoundError || t instanceof NoSuchMethodException) {
            return new DirectWebException("Error in tests dependencies. See Stacktrace for detail", t);
        }

        if (t instanceof NullPointerException || rootCause instanceof NullPointerException) {
            return new DirectWebException("Null pointer exception. Add assumption in test.", t);
        }

        return t;
    }
}
