package ru.yandex.autotests.direct.steps.utils;

import java.io.FileOutputStream;
import java.io.OutputStream;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.List;

import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;

import static ru.yandex.autotests.direct.steps.XlsTestProperties.getXlsTestProperties;

public class TableDiffer {
    private final XSSFCellStyle RED;
    private final XSSFCellStyle GREEN;

    private XSSFWorkbook workbook;

    public TableDiffer() {
        workbook = new XSSFWorkbook();
        RED = workbook.createCellStyle();
        GREEN = workbook.createCellStyle();
        RED.setFillForegroundColor(IndexedColors.PINK.index);
        RED.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        GREEN.setFillForegroundColor(IndexedColors.LIGHT_GREEN.index);
        GREEN.setFillPattern(FillPatternType.SOLID_FOREGROUND);
    }

    public Workbook getXlsxDiffFile(List<List<String>> expectedList, List<List<String>> actualList) {
        XSSFSheet expectedSheet = workbook.createSheet("expected");
        XSSFSheet actualSheet = workbook.createSheet("actual");
        XSSFSheet diffSheet = workbook.createSheet("diff");
        int maxHeight = Math.max(expectedList.size(), actualList.size());
        for (int i = 0; i < maxHeight; i++) {
            List<String> actualRowList = getSafely(actualList, i, Collections.emptyList());
            List<String> expectedRowList = getSafely(expectedList, i, Collections.emptyList());
            int maxWidth = Math.max(actualRowList.size(), expectedRowList.size());
            XSSFRow actualRow = actualSheet.createRow(i);
            XSSFRow expectedRow = expectedSheet.createRow(i);
            XSSFRow diffRow = diffSheet.createRow(i);
            for (int j = 0; j < maxWidth; j++) {
                XSSFCell actualCell = actualRow.createCell(j);
                XSSFCell expectedCell = expectedRow.createCell(j);
                XSSFCell diffCell = diffRow.createCell(j);
                String actualCellValue = getSafely(actualRowList, j, "");
                String expectedCellValue = getSafely(expectedRowList, j, "");
                actualCell.setCellValue(actualCellValue);
                expectedCell.setCellValue(expectedCellValue);

                if (actualCellValue.equals(expectedCellValue)) {
                    actualCell.setCellStyle(GREEN);
                    expectedCell.setCellStyle(GREEN);
                    diffCell.setCellValue(actualCellValue);
                    diffCell.setCellStyle(GREEN);
                } else {
                    actualCell.setCellStyle(RED);
                    expectedCell.setCellStyle(RED);
                    diffCell.setCellValue(
                            String.format("expected [%s] but was [%s]", expectedCellValue, actualCellValue));
                    diffCell.setCellStyle(RED);
                }
            }
        }
        return workbook;
    }

    public static void addLocalAttachment(Workbook wb, Logger log) {
        String fileName = Paths.get(getXlsTestProperties().getOutputDir(), System.nanoTime() + "_diff.xls")
                .toString();
        try (OutputStream out = new FileOutputStream(fileName)) {
            wb.write(out);
        } catch (Exception e) {
            log.error("Не удалось прикрепить файл различий", e);
        }
        log.info("Файл различий: " + fileName);
    }

    private <T> T getSafely(List<T> list, int index, T defaultValue) {
        if (index < list.size()) {
            return list.get(index);
        }
        return defaultValue;
    }
}
