package ru.yandex.autotests.directapi.banners.moderatebanners;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.banners.BannerFeatures;
import ru.yandex.autotests.directapi.banners.BannerLogins;
import ru.yandex.autotests.directapi.model.banners.CampaignBidsInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.allure.annotations.Title;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created by semkagtn on 11/7/14.
 * https://st.yandex-team.ru/TESTIRT-3249
 */
@Aqua.Test
@Features(BannerFeatures.MODERATE_BANNERS)
public class ModerateBannersNegativeTest {

    private static final String LOGIN = BannerLogins.CLIENT_SELF;

    protected LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private Long campaignId;
    private Long adId;

    @Before
    @Step("Подготовка данных для теста")
    public void createCampaign() {
        api.as(LOGIN);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        adId = api.userSteps.addCompletedAdGroupAndReturnAdId(campaignId);
    }

    @Test
    @Title("Несуществующая кампания")
    @ru.yandex.qatools.allure.annotations.TestCaseId("8377")
    public void nonexistentCampaign() {
        Long nonexistentCampaignID = campaignId + 77877;

        api.userSteps.shouldGetErrorOn(
                Method.MODERATE_BANNERS,
                new CampaignBidsInfoMap(api.type())
                        .withCampaignID(nonexistentCampaignID).getBean(),
                new AxisError(1));
    }

    @Test
    @Title("Удалённая кампания")
    @ru.yandex.qatools.allure.annotations.TestCaseId("8375")
    public void deletedCampaign() {
        api.userSteps.campaignSteps().campaignsDelete(campaignId);

        api.userSteps.shouldGetErrorOn(
                Method.MODERATE_BANNERS,
                new CampaignBidsInfoMap(api.type())
                        .withCampaignID(campaignId).getBean(),
                new AxisError(1));
    }

    @Test
    @Title("Чужая кампания с баннером")
    @ru.yandex.qatools.allure.annotations.TestCaseId("8378")
    public void campaignWhichDoesNotBelongToLoginWithBanner() {
        String otherLogin = BannerLogins.LOGIN_FOR_RUB;
        assumeThat("логины не равны", LOGIN, not(equalTo(otherLogin)));

        api.as(otherLogin);
        Long otherCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.addCompletedAdGroupAndReturnAdId(otherCampaignId);

        api.as(LOGIN);
        api.userSteps.shouldGetErrorOn(
                Method.MODERATE_BANNERS,
                new CampaignBidsInfoMap(api.type())
                        .withCampaignID(otherCampaignId).getBean(),
                new AxisError(54));
    }

    @Test
    @Title("Черновик и активный баннер в " + CampaignBidsInfoMap.BANNER_IDS)
    @ru.yandex.qatools.allure.annotations.TestCaseId("8379")
    public void draftBannerAndActiveBannerInBannerIDS() {
        Long activeBannerID = api.userSteps.addCompletedAdGroupAndReturnAdId(campaignId);
        api.userSteps.bannersFakeSteps().makeBannerActive(activeBannerID);

        api.userSteps.shouldGetErrorOn(
                Method.MODERATE_BANNERS,
                new CampaignBidsInfoMap(api.type())
                        .withBannerIDS(adId, activeBannerID).getBean(),
                new AxisError(155, AxisErrorDetails.BANNER_MUST_BE_DRAFT));
    }

    @Test
    @Title("Черновик и несуществующий баннер в " + CampaignBidsInfoMap.BANNER_IDS)
    @ru.yandex.qatools.allure.annotations.TestCaseId("8380")
    public void draftBannerAndNonexistentBannerInBannerIDS() {
        Long nonexistentBannerID = adId + 54329;

        api.userSteps.shouldGetErrorOn(
                Method.MODERATE_BANNERS,
                new CampaignBidsInfoMap(api.type())
                        .withBannerIDS(adId, nonexistentBannerID).getBean(),
                new AxisError(27, AxisErrorDetails.ADS_NOT_FOUND, nonexistentBannerID));
    }

    @Test
    @Title("Черновик и удалённый баннер в " + CampaignBidsInfoMap.BANNER_IDS)
    @ru.yandex.qatools.allure.annotations.TestCaseId("8376")
    public void draftBannerAndDeletedBannerInBannerIDS() {
        Long deletedBannerID = api.userSteps.addCompletedAdGroupAndReturnAdId(campaignId);
        api.userSteps.adsSteps().adsDelete(deletedBannerID);

        api.userSteps.shouldGetErrorOn(
                Method.MODERATE_BANNERS,
                new CampaignBidsInfoMap(api.type())
                        .withBannerIDS(adId, deletedBannerID).getBean(),
                new AxisError(27, AxisErrorDetails.ADS_NOT_FOUND, deletedBannerID));
    }

    @Test
    @Title("Свой баннер и чужой баннер в " + CampaignBidsInfoMap.BANNER_IDS)
    @ru.yandex.qatools.allure.annotations.TestCaseId("8381")
    public void loginBannerAndOtherBannerWhichDoesNotBelongToLogin() {
        String otherLogin = BannerLogins.LOGIN_FOR_RUB;
        assumeThat("логины не равны", LOGIN, not(equalTo(otherLogin)));

        api.as(otherLogin);
        Long otherCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        Long otherBannerID = api.userSteps.addCompletedAdGroupAndReturnAdId(otherCampaignId);

        api.as(LOGIN);
        api.userSteps.shouldGetErrorOn(
                Method.MODERATE_BANNERS,
                new CampaignBidsInfoMap(api.type())
                        .withBannerIDS(adId, otherBannerID).getBean(),
                new AxisError(54));
    }
}
