package ru.yandex.autotests.directapi.campaigns.geteventslog.campaign;

import java.util.Arrays;

import org.joda.time.DateTime;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.direct.utils.textresource.TextResourceFormatter;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.campaigns.CampaignFeatures;
import ru.yandex.autotests.directapi.campaigns.CampaignLogins;
import ru.yandex.autotests.directapi.common.api45.EventsLogItem;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.model.geteventslog.EventName;
import ru.yandex.autotests.directapi.model.geteventslog.EventType;
import ru.yandex.autotests.directapi.model.geteventslog.EventsLogItemMap;
import ru.yandex.autotests.directapi.model.geteventslog.GetEventsLogFilterMap;
import ru.yandex.autotests.directapi.model.geteventslog.GetEventsLogRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.not;
import static ru.yandex.autotests.direct.utils.matchers.BeanEquals.beanEquals;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Created by chicos on 06.02.14.<br/>
 * https://jira.yandex-team.ru/browse/TESTIRT-1400
 */
@Aqua.Test
@Features(CampaignFeatures.GET_EVENTS_LOG)
@Tag(TagDictionary.TRUNK)
public class GetCampaignFinishedEventsTest {

    protected LogSteps log = LogSteps.getLogger(this.getClass());

    private static final String CLIENT_LOGIN = CampaignLogins.SELF_CLIENT;
    private static final Currency CLIENT_CURRENCY = Currency.RUB;

    private static EventsLogItemMap finishedEvent;
    private static EventsLogItemMap cutDescriptionEvent;

    private static String timestamp1;
    private static String timestamp2;
    private static String timestampAfter;

    private static Long campaignId;
    private static String shortName = "Тестовая кампания API";
    private static Long campaignId2;
    private static String longName = "Тестовая кампания API длиной более 40 символов";

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(CLIENT_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

	@Rule
    public Trashman trasher = new Trashman(api);


    @BeforeClass
    public static void prepareEvents() {
        campaignId = api.userSteps.campaignSteps().addCampaign(
                new CampaignAddItemMap().defaultCampaignAddItem().withDefaultTextCampaign().withName(shortName));
        campaignId2 = api.userSteps.campaignSteps().addCampaign(
                new CampaignAddItemMap().defaultCampaignAddItem().withDefaultTextCampaign().withName(longName));

        timestamp1 = api.userSteps.changesSteps().getTimestamp();
        finishedEvent = new EventsLogItemMap(api.type())
                .campaignFinishedEvent(campaignId.intValue(), DateTime.now().toDate());
        api.userSteps.eventLogFakeSteps().addEvents(finishedEvent);
        finishedEvent.withEventName(TextResourceFormatter.resource(EventName.CAMPAIGN_FINISHED).toString())
                .withTimestamp(null);

        timestamp2 = api.userSteps.changesSteps().getTimestamp();

        cutDescriptionEvent = new EventsLogItemMap(api.type())
                .campaignFinishedEvent(campaignId2.intValue(), DateTime.now().toDate());
        api.userSteps.eventLogFakeSteps().addEvents(cutDescriptionEvent);
        cutDescriptionEvent.withEventName(TextResourceFormatter.resource(EventName.CAMPAIGN_FINISHED).toString())
                .withTimestamp(null);

        timestampAfter = api.userSteps.changesSteps().getTimestamp();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5555")
    public void getEventByEmptyFilter() {
        EventsLogItem[] events = api.userSteps.getEventsLogSteps().getEventsLog(
                new GetEventsLogRequestMap(api.type())
                        .withTimestampFrom(timestamp1)
                        .withCurrency(CLIENT_CURRENCY)
        );

        finishedEvent.withTextDescription(null);
        assertThat("присутствует событие", Arrays.asList(events),
                hasItem(beanEquals((EventsLogItem) finishedEvent.getBean())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5552")
    public void getEventByMoneyOutType() {
        EventsLogItem[] events = api.userSteps.getEventsLogSteps().getEventsLog(
                new GetEventsLogRequestMap(api.type())
                        .withTimestampFrom(timestamp1)
                        .withTimestampTo(timestamp2)
                        .withCurrency(CLIENT_CURRENCY)
                        .withFilter(new GetEventsLogFilterMap(api.type())
                                .withEventType(EventType.MONEYOUT))
        );

        finishedEvent.withTextDescription(null);
        assertThat("отсутствует некоторое событие", Arrays.asList(events),
                not(hasItem(beanEquals((EventsLogItem) finishedEvent.getBean()))));
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5551")
    public void getEventByCampaignFinishedType() {
        EventsLogItem[] events = api.userSteps.getEventsLogSteps().getEventsLog(
                new GetEventsLogRequestMap(api.type())
                        .withTimestampFrom(timestamp1)
                        .withTimestampTo(timestamp2)
                        .withCurrency(CLIENT_CURRENCY)
                        .withFilter(new GetEventsLogFilterMap(api.type())
                                .withEventType(EventType.CAMPAIGNFINISHED))
        );

        finishedEvent.withTextDescription(null);
        assertThat("присутствует событие", Arrays.asList(events),
                hasItem(beanEquals((EventsLogItem) finishedEvent.getBean())));
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5556")
    public void getEventByCampaignId() {
        EventsLogItem[] events = api.userSteps.getEventsLogSteps().getEventsLog(
                new GetEventsLogRequestMap(api.type())
                        .withTimestampFrom(timestamp1)
                        .withTimestampTo(timestamp2)
                        .withCurrency(CLIENT_CURRENCY)
                        .withFilter(new GetEventsLogFilterMap(api.type())
                                .withCampaignIDS(campaignId.intValue()))
        );

        finishedEvent.withTextDescription(null);
        assertThat("присутствует событие", Arrays.asList(events),
                hasItem(beanEquals((EventsLogItem) finishedEvent.getBean())));
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5557")
    public void getEventWithTextDescription() {
        EventsLogItem[] events = api.userSteps.getEventsLogSteps().getEventsLog(
                new GetEventsLogRequestMap(api.type())
                        .withTimestampFrom(timestamp1)
                        .withTimestampTo(timestamp2)
                        .withCurrency(CLIENT_CURRENCY)
                        .withWithTextDescription(Value.YES)
                        .withFilter(new GetEventsLogFilterMap(api.type())
                                .withEventType(EventType.CAMPAIGNFINISHED))
        );

        finishedEvent.withTextDescription(shortName);
        assertThat("присутствует событие", Arrays.asList(events),
                hasItem(beanEquals((EventsLogItem) finishedEvent.getBean())));
    }

    @Test
    @Description("Длинное описание события должно быть обрезано до 40 символов плюс многоточие в конце. " +
            "Необходимо для корректного отображения push-уведомлений.")
    @ru.yandex.qatools.allure.annotations.TestCaseId("5553")
    public void cutEventDescriptionTo40Symbols() {
        EventsLogItem[] events = api.userSteps.getEventsLogSteps().getEventsLog(
                new GetEventsLogRequestMap(api.type())
                        .withTimestampFrom(timestamp2)
                        .withTimestampTo(timestampAfter)
                        .withCurrency(CLIENT_CURRENCY)
                        .withWithTextDescription(Value.YES)
                        .withFilter(new GetEventsLogFilterMap(api.type())
                                .withEventType(EventType.CAMPAIGNFINISHED))
        );

        cutDescriptionEvent.withTextDescription(longName.substring(0, 40) + "...");
        assertThat("присутствует событие", Arrays.asList(events),
                hasItem(beanEquals((EventsLogItem) cutDescriptionEvent.getBean())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5554")
    public void getEventWithLastEventsOnly() {
        EventsLogItem[] events = api.userSteps.getEventsLogSteps().getEventsLog(
                new GetEventsLogRequestMap(api.type())
                        .withTimestampFrom(timestamp1)
                        .withCurrency(CLIENT_CURRENCY)
                        .withFilter(new GetEventsLogFilterMap(api.type())
                                .withEventType(EventType.CAMPAIGNFINISHED))
        );

        finishedEvent.withTextDescription(null);
        assertThat("присутствует событие", Arrays.asList(events),
                hasItem(beanEquals((EventsLogItem) finishedEvent.getBean())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5558")
    public void getEventWithTimestampAfterEvent() {
        EventsLogItem[] events = api.userSteps.getEventsLogSteps().getEventsLog(
                new GetEventsLogRequestMap(api.type())
                        .withTimestampFrom(timestamp2)
                        .withCurrency(CLIENT_CURRENCY)
        );

        finishedEvent.withTextDescription(null);
        assertThat("отсутствует некоторое событие", Arrays.asList(events),
                not(hasItem(beanEquals((EventsLogItem) finishedEvent.getBean()))));
    }
}
