package ru.yandex.autotests.directapi.campaigns.geteventslog.currency;

import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.campaigns.CampaignFeatures;
import ru.yandex.autotests.directapi.common.api45.ClientInfo;
import ru.yandex.autotests.directapi.common.api45.EventsLogItem;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.model.geteventslog.EventsLogItemMap;
import ru.yandex.autotests.directapi.model.geteventslog.GetEventsLogFilterMap;
import ru.yandex.autotests.directapi.model.geteventslog.GetEventsLogRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.everyItem;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

/**
 * Migrated from old school project (api45)
 * <p>
 * Created by chicos on 31.01.14.
 * https://jira.yandex-team.ru/browse/TESTIRT-1400
 */

@Aqua.Test
@Features(CampaignFeatures.GET_EVENTS_LOG)
@Stories(ApiStories.CURRENCY)
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class GetEventsLogPayedCurrencyTest {
    protected LogSteps log = LogSteps.getLogger(this.getClass());

    private String clientCurrency;
    private String beforeTime;
    private Long campaignId;

    @Parameterized.Parameter
    public String clientLogin;

    @Parameterized.Parameter(1)
    public Currency currency;

    @Parameterized.Parameter(2)
    public float sum;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void initCurrencyRates() {
        api.userSteps.getDarkSideSteps().getRunScriptSteps().runFillCurrencyRates();
    }

    @Before
    public void prepareEvent() {
        campaignId = api.as(clientLogin).userSteps.campaignSteps().addDefaultTextCampaign();

        String[] currencies = ((ClientInfo) api.userSteps.clientSteps().getClientInfo(clientLogin)).getClientCurrencies();
        clientCurrency = (currencies != null && currencies.length > 0) ? (currencies[0]) : (null);

        beforeTime = api.userSteps.changesSteps().getTimestamp();
        EventsLogItemMap event = new EventsLogItemMap(api.type()).moneyInEvent(campaignId.intValue(), sum, clientCurrency);
        api.userSteps.eventLogFakeSteps().addEvents(event);
    }

    @Parameterized.Parameters(name = "login = {0}, request currency = {1}, payed = {2}")
    public static Collection currencies() {
        Object[][] data = new Object[][]{
                {Logins.LOGIN_FOR_RUB, Currency.RUB, 300f},
                {Logins.LOGIN_FOR_USD, Currency.USD, 10f},
                {Logins.CLIENT_BYN, Currency.BYN, 10f}
        };
        return Arrays.asList(data);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5572")
    public void getEventsLogPayedFiled() {
        EventsLogItem[] events = api.as(clientLogin).userSteps.getEventsLogSteps().getEventsLog(
                new GetEventsLogRequestMap(api.type())
                        .withTimestampFrom(beforeTime)
                        .withCurrency(currency)
                        .withFilter(new GetEventsLogFilterMap(api.type())
                                .withCampaignIDS(campaignId.intValue()))
                        .withWithTextDescription(Value.YES)
        );
        List<Float> payeds = extract(events, on(EventsLogItem.class).getAttributes().getPayed());
        assertThat("корректное значение Payed в ответе", payeds,
                everyItem(equalTo(Money.valueOf(sum, clientCurrency).subtractVAT().convert(currency).setScale(2, RoundingMode.HALF_UP).floatValue())));
    }
}
