package ru.yandex.autotests.directapi.clients.getclientinfo;

import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.clients.ClientFeatures;
import ru.yandex.autotests.directapi.clients.ClientLogins;
import ru.yandex.autotests.directapi.common.api45mng.CreateNewSubclientResponse;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.balanceclient.NotifyClient2JSONRequest;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.ClientFakeInfo;
import ru.yandex.autotests.directapi.darkside.steps.DarkSideSteps;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.clients.ClientInfoMap;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.beandiffer.beanconstraint.WhiteListConstraint;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsNull.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by chicos on 29.05.2015.
 */
@Aqua.Test
@Features(ClientFeatures.GET_CLIENT)
@Stories(ApiStories.CURRENCY)
@RunWith(Parameterized.class)
@Issue("https://st.yandex-team.ru/DIRECT-41617")
@Description("Проверка значений овердрафта, возвращаемых в валюте клиента в поле OverdraftSumAvailableInCurrency")
public class GetClientInfoOverdraftSumInCurrencyTest {
    protected LogSteps log = LogSteps.getLogger(this.getClass());

    private DarkSideSteps darkSideSteps = new DarkSideSteps();

    private static String clientYE;
    private static final Class clientWsdl = ru.yandex.autotests.directapi.common.api45.APIPort_PortType.class;
    private static final Class managerWsdl = ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType.class;

    private static String singleClientRUB = ClientLogins.SINGLE_CLIENT_RUB;
    private static String srvClientRUB = ClientLogins.SRV_CLIENT_RUB;
    private static String singleClientBYN = ClientLogins.CLIENT_BYN;
    private Money overdraftLimit;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public String operator;

    @Parameterized.Parameter(value = 2)
    public String client;

    @Parameterized.Parameter(value = 3)
    public Currency currency;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection logins() {
        prepareClient();
        Object[][] data = new Object[][]{
                {"Клиент RUB", singleClientRUB, singleClientRUB, Currency.RUB},
                {"Менеджер сервисируемого клиента YE", Logins.MANAGER_GEO, clientYE, Currency.YND_FIXED},
                {"Менеджер сервисируемого клиента RUB", Logins.MANAGER_DEFAULT, srvClientRUB, Currency.RUB},
                {"Клиент BYN", singleClientBYN, singleClientBYN, Currency.BYN},
        };
        return Arrays.asList(data);
    }

    @Rule
    public Trashman trasher = new Trashman(api);

    private static void prepareClient() {
        api.as(Logins.MANAGER_GEO).wsdl(managerWsdl);
        clientYE = ((CreateNewSubclientResponse) api.userSteps.clientSteps()
                .createNewServicedSubClient("subClient", Logins.MANAGER_GEO)).getLogin();
        assumeThat("создали сервисируемого клиента", clientYE, notNullValue());
        api.userSteps.clientFakeSteps().setAPIEnabled(clientYE, Value.YES);
    }

    @Before
    public void prepareOverdraft() {
        api.as(operator).wsdl(clientWsdl);
        ClientFakeInfo fakeInfo = api.userSteps.clientFakeSteps().getClientData(client);
        overdraftLimit = Money.valueOf(MoneyCurrency.get(currency).getMinTransferAmount().floatValue(), currency);
        NotifyClient2JSONRequest request = new NotifyClient2JSONRequest().defaultParams()
                .withClientID(Long.valueOf(fakeInfo.getClientID()))
                .withTimestamp()
                .withOverdraftLimit(overdraftLimit.floatValue())
                .withOverdraftSpent(0f)
                .withClientCurrency(currency.value());

        log.info("Вызываем метод NotifyClient2 - выставим значение овердрафта клиенту " + client);
        darkSideSteps.getBalanceClientNotifyClientJsonSteps().notifyClientNoErrors(request);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5766")
    public void overdraftInCurrencyTest() {
        Object clientInfo = api.userSteps.clientSteps().getClientInfo(client);
        ClientInfoMap expectedInfo = new ClientInfoMap(api.type())
                .withOverdraftSumAvailable(null)
                .withOverdraftSumAvailableInCurrency(
                        overdraftLimit.subtractVAT().setScale(2, RoundingMode.DOWN).floatValue());

        assertThat("корректные суммы овердрафта", clientInfo, beanEquivalent(expectedInfo.getBean())
                .fields(new WhiteListConstraint().putFields(
                        ClientInfoMap.OVERDRAFT_SUM_AVAILABLE,
                        ClientInfoMap.OVERDRAFT_SUM_AVAILABLE_IN_CURRENCY)));
    }
}
