package ru.yandex.autotests.directapi.clients.search;

import org.apache.commons.lang.RandomStringUtils;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.autotests.directapi.clients.ClientFeatures;
import ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45mng.ClientInfo;
import ru.yandex.autotests.directapi.common.api45mng.SearchClientsResponse;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.clients.SearchClientsRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.core.IsEqual.equalTo;
import static ru.yandex.autotests.directapi.model.clients.SearchClientsRequestMap.*;

/**
 * User: mariabye
 * Date: 16.08.12
 *
 * http://wiki.yandex-team.ru/Direkt/API/ExtraFeatures#metodsearchclients
 */
@Aqua.Test(title = "SearchClients: поиск по одному полю")
@RunWith(Parameterized.class)
@Tag(TagDictionary.TRUNK)
@Features(ClientFeatures.SEARCH_CLIENTS)
public class SearchClientsRequestTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class).as(Logins.SUPER_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameters(name = "Search request = {0}")
    public static Collection roleValues() {
        ClientInfo clientToVerify = api.userSteps.clientSteps()
                .getClientInfo(Logins.CLIENT_FREE_YE_DEFAULT);

        ClientInfo dotLogin = api.as(Logins.SUPER_LOGIN).userSteps.clientSteps()
                .getClientInfo(Logins.CLIENT_DOT_LOGIN);

        Object[][] testData = new Object[][]{
                {
                        new SearchClientsRequestMap(api.type()).put(LOGIN, RandomStringUtils.randomAlphabetic(257)),
                        hasSize(0)
                },
                {
                        new SearchClientsRequestMap(api.type()).put(LOGIN, Logins.AGENCY_YE_DEFAULT),
                        Matchers.hasItem(CoreMatchers.equalTo(Logins.AGENCY_YE_DEFAULT))
                },
                {
                        new SearchClientsRequestMap(api.type()).put(LOGIN, Logins.MANAGER_DEFAULT),
                        Matchers.hasItem(CoreMatchers.equalTo(Logins.MANAGER_DEFAULT))
                },
                {
                        new SearchClientsRequestMap(api.type()).put(USER_ID, clientToVerify.getUserID()),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type()).put(USER_ID, clientToVerify.getUserID()/100),
                        hasSize(0)
                },
                {
                        new SearchClientsRequestMap(api.type()).put(CLIENT_ID, clientToVerify.getClientID()),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type()).put(CLIENT_ID, clientToVerify.getClientID()/100),
                        hasSize(0)
                },
                {
                        new SearchClientsRequestMap(api.type()).put(LOGIN, clientToVerify.getLogin()),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type()).put(LOGIN, dotLogin.getLogin()),
                        Matchers.hasItem(equalTo(dotLogin.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type()).put(LOGIN, dotLogin.getLogin().replaceAll("\\.","-")),
                        Matchers.hasItem(equalTo(dotLogin.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type())
                        .withLogin(
                                        clientToVerify.getLogin().substring(0, clientToVerify.getLogin().length() - 2)),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type())
                                .withName(clientToVerify.getFIO())
                                .withLogin(clientToVerify.getLogin()),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type())
                                .withManagerLogins(new String[]{Logins.MANAGER_DEFAULT}),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type())
                                .withManagerLogins(new String[]{Logins.MANAGER_DEFAULT.replaceAll("-",".")}),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type())
                                .withLogin(clientToVerify.getLogin())
                                .withName(clientToVerify.getFIO())
                                .withUserID(clientToVerify.getUserID())
                                .withClientID(clientToVerify.getClientID())
                                .withManagerLogins(Logins.MANAGER_DEFAULT),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type())
                                .withLogin(clientToVerify.getLogin())
                                .withManagerLogins(new String[]{Logins.MANAGER_DEFAULT}),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type())
                                .withName(clientToVerify.getFIO())
                                .withManagerLogins(new String[]{Logins.MANAGER_DEFAULT}),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type())
                                .withLogin(clientToVerify.getLogin())
                                .withManagerLogins(managerLogins),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type())
                                .withName(clientToVerify.getFIO())
                                .withManagerLogins(managerLogins),
                        Matchers.hasItem(equalTo(clientToVerify.getLogin()))
                },
                {
                        new SearchClientsRequestMap(api.type())
                                .withName(String.format(" %s ", clientToVerify.getFIO())),
                        hasSize(0)
                },
                {
                        new SearchClientsRequestMap(api.type())
                                .withLogin( clientToVerify.getLogin())
                                .withName(clientToVerify.getFIO()+".."),
                        hasSize(0)
                },
                {
                        new SearchClientsRequestMap(api.type())
                                .withLogin(clientToVerify.getLogin())
                                .withManagerLogins(new String[]{managerLogins[0]}),
                        hasSize(0)
                },
        };
        return Arrays.asList(testData);
    }

    @Parameterized.Parameter(value = 0)
    public SearchClientsRequestMap searchClientsRequestMap;

    @Parameterized.Parameter(value = 1)
    public Matcher responseMatch;

    private static String[] managerLogins = new String[]{"at-direct-transfer-mngr","at-direct-mngr-full"};

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5830")
    public void searchClient() {
        log.info("Поиска клиента");
        log.info(JsonUtils.toString(searchClientsRequestMap.getBean()));
        SearchClientsResponse result = api.userSteps.clientSteps().searchClients(searchClientsRequestMap);
        log.info("Проверка результатов поиска");
        List<String> logins = extract(result.getClients(), on(ClientInfo.class).getLogin());
        assertThat("Не получен клиент, удовлетворяющий запросу поиска", logins, responseMatch);
        log.info("Искомый клиент получен");
    }


}

