package ru.yandex.autotests.directapi.finance.accountmanagement;


import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.common.api45.CreateNewSubclientResponse;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.EnableSharedAccountRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;


/**
 * User: mariabye
 * Date: 12.11.13
 * <p>
 * https://jira.yandex-team.ru/browse/DIRECT-25909
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT)
public class DisabledForAccountThatTurnedOffTest {

    public static String agencyLogin = FinanceLogins.AGENCY_ENABLE_SHARED_ACCOUNT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(agencyLogin);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static Account account;
    private static String subClientLogin;

    @BeforeClass
    public static void sharedAccountFlagOn() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(agencyLogin);
        api.userSteps.clientFakeSteps().enableToCreateSubClients(agencyLogin);
        CreateNewSubclientResponse subClient = api.userSteps.clientSteps()
                .createNewAgencySubClient("create", agencyLogin, Currency.RUB);
        subClientLogin = subClient.getLogin();
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5935")
    public void invoiceForAccountIDAllCampaignsDeleted() {
        long campaignID = api.clientLogin(subClientLogin).userSteps.campaignSteps().addDefaultTextCampaign();
        account = api.userSteps.financeSteps().getAccount(subClientLogin);
        api.userSteps.campaignSteps().campaignsDelete(campaignID);

        api.userSteps.financeSteps()
                .shouldGetErrorsOnAMInvoice(account.getAccountID(),
                        400f, Currency.RUB,
                        new AxisError(515, AxisErrorDetails.OPERATIONS_WITH_SHARED_ACCOUNT_DISABLED,
                                account.getAccountID()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5937")
    public void depositForAccountIDAllCampaignsDeleted() {
        long campaignID = api.clientLogin(subClientLogin).userSteps.campaignSteps().addDefaultTextCampaign();
        account = api.userSteps.financeSteps().getAccount(subClientLogin);
        api.userSteps.campaignSteps().campaignsDelete(campaignID);

        api.userSteps.financeSteps()
                .shouldGetErrorsOnAMDeposit(
                        new PaymentMap[]{
                                new PaymentMap(api.type())
                                        .withContract(User.get(agencyLogin).getContractID())
                                        .withAccountID(account.getAccountID())
                                        .withCurrency(Currency.RUB)
                                        .withAmount(400f)},
                        new AxisError(515, AxisErrorDetails.OPERATIONS_WITH_SHARED_ACCOUNT_DISABLED,
                                account.getAccountID()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5939")
    public void getByLoginWithoutCampaignsAndAccount() {
        long campaignID = api.clientLogin(subClientLogin).userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignSteps().campaignsDelete(campaignID);

        api.userSteps.financeSteps()
                .shouldGetErrorsOnAccountManagementGet(
                        subClientLogin,
                        new AxisError(515, AxisErrorDetails.EMPTY_STRING));
    }

    //Чтобы не подбирали accountID - отдаем "нет прав", если счет по заданному accountID не включен
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5942")
    public void getByAccountIdForAccountAllCampaignsDeleted() {
        long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(subClientLogin);
        account = api.userSteps.financeSteps().getAccount(subClientLogin);
        api.userSteps.campaignSteps().campaignsDelete(subClientLogin, campaignID);

        api.userSteps.financeSteps()
                .shouldGetErrorsOnAccountManagementGet(
                        account.getAccountID(),
                        new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5943")
    public void enableSharedAccountForAccountAllCampaignsDeleted() {
        long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(subClientLogin);
        api.userSteps.campaignSteps().campaignsDelete(subClientLogin, campaignID);

        api.userSteps.financeSteps().shouldGetErrorsOnEnableSharedAccount(
                new EnableSharedAccountRequestMap(api.type()).withLogins(subClientLogin),
                new AxisError(520,
                        AxisErrorDetails.TO_ENABLE_SHARED_ACCOUNT_AT_LEAST_ONE_CAMPAIGN_REQUIRED));
    }

    @Test
    public void enableSharedAccountForTurnedOffAccount() {
        long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign(subClientLogin);

        int shard = api.userSteps.clientFakeSteps().getUserShard(subClientLogin);
        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard);

        CampaignsRecord campaiangs = jooqDbSteps.campaignsSteps().getCampaignById(campaignId);
        campaiangs.setWalletCid(0L);
        jooqDbSteps.campaignsSteps().updateCampaigns(campaiangs);

        api.userSteps.financeSteps().shouldGetErrorsOnEnableSharedAccount(
                new EnableSharedAccountRequestMap(api.type()).withLogins(subClientLogin),
                new AxisError(519,
                        AxisErrorDetails.ACCOUNT_ENABLING_POSSIBLE_NOT_EARLIER_THAN_ONE_DAY_SINCE_LAST_TRY));
    }
}
