package ru.yandex.autotests.directapi.finance.accountmanagement.checkpayment;

import java.util.Arrays;

import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Action;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayCampaignsByCardInfoMap;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.AccountManagementRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;
import static org.apache.commons.lang3.StringUtils.capitalize;

/**
 * Created by semkagtn on 22.09.15.
 * https://st.yandex-team.ru/TESTIRT-6408
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_CHECK_PAYMENT)
@Description("Проверка валидации поля CustomTransactionId в запросе: негативные проверки")
@Issue("https://st.yandex-team.ru/DIRECT-42633")
@RunWith(Parameterized.class)
public class CheckPaymentCustomTransactionIdValidationNegativeTest {

    private static final String LOGIN = Logins.CLIENT_FREE_YE_DEFAULT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(LOGIN);

    @Rule
    public Trashman trasher = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public String description;

    @Parameterized.Parameter(1)
    public String customTransactionId;

    @Parameterized.Parameter(2)
    public AxisError expectedError;

    @Parameterized.Parameters(name = "{0}: CustomTransactionID = {1}")
    public static Iterable<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {"число символов меньше требуемого",
                        randomAlphanumeric(PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID_LENGTH - 1),
                        new AxisError(71, AxisErrorDetails.FIELD_MUST_CONTAINS_N_ALPHANUMERIC_CHARS,
                                capitalize(PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID),
                                PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID_LENGTH)},

                {"число символов больше требуемового",
                        randomAlphanumeric(PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID_LENGTH + 1),
                        new AxisError(71, AxisErrorDetails.THE_NUMBER_OF_CHARACTERS_CANNOT_EXCEED,
                                capitalize(PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID),
                                PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID_LENGTH)},

                {"запрещённые символы в запросе",
                        randomAlphanumeric(PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID_LENGTH - 1) + 'щ',
                        new AxisError(71, AxisErrorDetails.FIELD_MUST_CONTAINS_N_ALPHANUMERIC_CHARS,
                                capitalize(PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID),
                                PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID_LENGTH)},

                {"отутствие значения", null,
                        new AxisError(71, AxisErrorDetails.FIELD_MUST_NOT_BE_EMPTY,
                                capitalize(PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID))},

                {"строка из пробела", " ",
                        new AxisError(71, AxisErrorDetails.STRING_FIELD_SHOULD_NOT_BE_EMPTY,
                                capitalize(PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID))},

                {"пустая строка", "",
                        new AxisError(71, AxisErrorDetails.STRING_FIELD_SHOULD_NOT_BE_EMPTY,
                                capitalize(PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID))},

                {"строка правильной длины с пробелами в начале и в конце",
                        " " + randomAlphanumeric(PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID_LENGTH) + " ",
                        new AxisError(71, AxisErrorDetails.THE_NUMBER_OF_CHARACTERS_CANNOT_EXCEED,
                                capitalize(PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID),
                                PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID_LENGTH)},
        });
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5956")
    public void invalidCustomTransactionIdInRequest() {
        api.userSteps.financeSteps().shouldGetErrorOn(Method.ACCOUNT_MANAGEMENT,
                new AccountManagementRequestMap<>(api.type())
                        .withAction(Action.CHECK_PAYMENT)
                        .withCustomTransactionId(customTransactionId),
                expectedError);
    }
}
