package ru.yandex.autotests.directapi.finance.accountmanagement.deposit;

import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PayMethod;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * User: mariabye
 * Date: 17.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_DEPOSIT)
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class AccountIDTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    static String agency1Login = FinanceLogins.AGENCY_DEPOSIT_1;
    static String agency2Login = FinanceLogins.AGENCY_DEPOSIT_2;
    static String contractID = User.get(agency1Login).getExternalContractID();

    static int accountID1;
    static int accountNoRightID;
    static int accountSelfNoRightID;

    @Parameterized.Parameter
    public PaymentMap paymentMap;

    @Parameterized.Parameters(name = "Payment = {0}")
    public static Collection payments() {
        Float minPaymentAmount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT().floatValue();
        Object[][] data = new Object[][]{
                {new PaymentMap(api.type()).withContract(contractID).withAmount(minPaymentAmount)},
                {new PaymentMap(api.type()).withOrigin(PayMethod.OVERDRAFT).withAmount(minPaymentAmount)},
                {new PaymentMap(api.type()).withOrigin(PayMethod.YM).withAmount(minPaymentAmount)}
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void initTest() {
        api.as(FinanceLogins.INVOICE_FREE_CLIENT).userSteps.addModeratedCampaign();
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(FinanceLogins.INVOICE_FREE_CLIENT);
        accountSelfNoRightID = api.as(FinanceLogins.INVOICE_FREE_CLIENT).userSteps.financeSteps()
                .getAccountID(FinanceLogins.INVOICE_FREE_CLIENT);
        api.as(agency1Login);
        api.userSteps.clientFakeSteps().enableAgencyWallet(agency1Login);
        api.userSteps.createAccount(FinanceLogins.DEPOSIT_SC_1);
        accountID1 = api.userSteps.financeSteps().getAccountID(FinanceLogins.DEPOSIT_SC_1);
        accountNoRightID = api.userSteps.financeSteps().getAccountID(FinanceLogins.DEPOSIT_SC_2);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5958")
    public void set0AccountID() {
        paymentMap.withAccountID(0);
        api.as(agency1Login).userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new PaymentMap[]{paymentMap},
                new AxisError(54, AxisErrorDetails.EMPTY_STRING));

    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5959")
    public void setNegativeAccountID() {
        paymentMap.withAccountID(-accountID1);
        api.as(agency1Login).userSteps
                .financeSteps().shouldGetErrorOnAMDeposit(
                new AxisError(71,
                        AxisErrorDetails.FIELD_SHOULD_CONTAIN_AN_UNSIGNED_INTEGER_VALUE,
                        "Payments[0].AccountID"),
                paymentMap);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5960")
    public void setSelfNoRightAccountID() {
        paymentMap.withAccountID(accountSelfNoRightID);
        api.as(agency1Login).userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new PaymentMap[]{paymentMap},
                new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5961")
    public void setAgencyNoRightAccountID() {
        paymentMap.withAccountID(accountNoRightID);
        api.as(agency2Login).userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new PaymentMap[]{paymentMap},
                new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5962")
    public void setCampaignID() {
        long campaignID = api.as(agency1Login).userSteps.campaignSteps()
                .addDefaultTextCampaign(FinanceLogins.DEPOSIT_SC_1);
        paymentMap.withAccountID(campaignID);
        api.as(agency1Login).userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new PaymentMap[]{paymentMap},
                new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }
}
