package ru.yandex.autotests.directapi.finance.accountmanagement.deposit;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.model.ShardNumbers;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;

/**
 * User: pavryabov
 * Date: 27.01.14
 * <p/>
 * https://jira.yandex-team.ru/browse/TESTIRT-1429
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_DEPOSIT)
@RunWith(Parameterized.class)
public class AccountManagementDepositMassShardingTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public String operator;

    @Parameterized.Parameter(value = 1)
    public String clientFirstShard;

    @Parameterized.Parameter(value = 2)
    public String clientSecondShard;

    @Parameterized.Parameters(name = "Operator = {0}, ClientFirstShard = {1}, ClientSecondShard = {2}")
    public static Collection strategies() {
        Object[][] data = new Object[][]{
                {Logins.AGENCY_SHARD_2, Logins.ACCOUNT_SUBCLIENT_SHARD1_3, Logins.ACCOUNT_SUBCLIENT_SHARD_3}
        };
        return Arrays.asList(data);
    }

    public Account accountFirstShard;
    public Account accountSecondShard;
    public Currency currency = null;
    public Money amount;

    @BeforeClass
    public static void reshard() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(Logins.AGENCY_SHARD_2);
        api.as(Logins.AGENCY_SHARD_2).userSteps.createAccount(Logins.ACCOUNT_SUBCLIENT_SHARD_3);
        api.userSteps.createAccount(Logins.ACCOUNT_SUBCLIENT_SHARD1_3);
        api.userSteps.clientFakeSteps().reshardUserWithLock(Logins.AGENCY_SHARD_2, ShardNumbers.EXTRA_SHARD);
        api.userSteps.clientFakeSteps().reshardUserWithLock(Logins.ACCOUNT_SUBCLIENT_SHARD_3, ShardNumbers.EXTRA_SHARD);
        api.userSteps.clientFakeSteps()
                .reshardUserWithLock(Logins.ACCOUNT_SUBCLIENT_SHARD1_3, ShardNumbers.DEFAULT_SHARD);
    }

    @Before
    public void initTest() {
        accountFirstShard = api.as(operator).userSteps.financeSteps().getAccount(clientFirstShard);
        accountSecondShard = api.userSteps.financeSteps().getAccount(clientSecondShard);
        api.userSteps.balanceSteps().synchronizeWithBalance(accountFirstShard.getAccountID());
        api.userSteps.balanceSteps().synchronizeWithBalance(accountSecondShard.getAccountID());
        api.userSteps.financeSteps().verifyCampaignExistsInBalance(accountFirstShard.getAccountID());
        api.userSteps.financeSteps().verifyCampaignExistsInBalance(accountSecondShard.getAccountID());
        currency = Currency.getFor(accountFirstShard.getCurrency());
        amount = MoneyCurrency.get(currency).getMinInvoiceAmount();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5963")
    public void accountManagementDepositTest() {
        api.userSteps.financeSteps().deposit(
                new PaymentMap(api.type())
                        .withAccountID(accountFirstShard.getAccountID())
                        .withCurrency(currency)
                        .withAmount(amount.addVAT().floatValue())
                        .withContract(
                                api.userSteps.balanceSteps().getDefaultContract(User.get(operator).getClientID())),
                new PaymentMap(api.type())
                        .withAccountID(accountSecondShard.getAccountID())
                        .withCurrency(currency)
                        .withAmount(amount.addVAT().floatValue())
                        .withContract(api.userSteps.balanceSteps().getDefaultContract(User.get(operator).getClientID()))
        );

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps()
                        .accountAmountChangedOnSum(accountFirstShard, amount.floatValue()), equalTo(true));

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps()
                        .accountAmountChangedOnSum(accountSecondShard, amount.floatValue()), equalTo(true));
    }
}
