package ru.yandex.autotests.directapi.finance.accountmanagement.deposit;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.model.ShardNumbers;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;

/**
 * User: pavryabov
 * Date: 22.01.14
 * <p>
 * https://jira.yandex-team.ru/browse/TESTIRT-1429
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_DEPOSIT)
@RunWith(Parameterized.class)
public class AccountManagementDepositShardingPositiveTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public String operator;

    @Parameterized.Parameter(value = 1)
    public String client;

    @Parameterized.Parameter(value = 2)
    public String accountOwner;

    @Parameterized.Parameters(name = "Operator = {0}, Client = {1}")
    public static Collection strategies() {
        Object[][] data = new Object[][]{
                {Logins.AGENCY_SHARD_2, Logins.ACCOUNT_SUBCLIENT_SHARD_2, Logins.AGENCY_SHARD_2},
                {Logins.AGENCY_SHARD_2, Logins.ACCOUNT_SUBCLIENT_SHARD1_2, Logins.AGENCY_SHARD_2},
        };
        return Arrays.asList(data);
    }

    public Account account;
    public Currency currency = null;
    public Money amount;

    @BeforeClass
    public static void reshard() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(Logins.AGENCY_YE_DEFAULT);
        api.as(Logins.AGENCY_YE_DEFAULT).userSteps.createAccount(Logins.ACCOUNT_SUBCLIENT_SHARD2_2);
        api.userSteps.clientFakeSteps().enableAgencyWallet(Logins.AGENCY_SHARD_2);
        api.as(Logins.AGENCY_SHARD_2).userSteps.createAccount(Logins.ACCOUNT_SUBCLIENT_SHARD_2);
        api.userSteps.createAccount(Logins.ACCOUNT_SUBCLIENT_SHARD1_2);
        api.userSteps.clientFakeSteps().reshardUserWithLock(Logins.AGENCY_SHARD_2, ShardNumbers.EXTRA_SHARD);
        api.userSteps.clientFakeSteps().reshardUserWithLock(Logins.ACCOUNT_SUBCLIENT_SHARD_2, ShardNumbers.EXTRA_SHARD);
        api.userSteps.clientFakeSteps()
                .reshardUserWithLock(Logins.ACCOUNT_SUBCLIENT_SHARD1_2, ShardNumbers.DEFAULT_SHARD);
        api.userSteps.clientFakeSteps()
                .reshardUserWithLock(Logins.ACCOUNT_SUBCLIENT_SHARD2_2, ShardNumbers.EXTRA_SHARD);
    }

    @Before
    public void initTest() {
        account = api.as(accountOwner).userSteps.financeSteps().getAccount(client);
        api.userSteps.financeSteps().verifyCampaignExistsInBalance(account.getAccountID());
        api.userSteps.balanceSteps().synchronizeWithBalance(account.getAccountID());
        currency = Currency.getFor(account.getCurrency());
        amount = MoneyCurrency.get(currency).getMinInvoiceAmount();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5965")
    public void accountManagementDepositTest() {
        api.as(operator).userSteps.financeSteps().deposit(
                new PaymentMap(api.type())
                        .withAccountID(account.getAccountID())
                        .withCurrency(currency)
                        .withAmount(amount.addVAT().floatValue())
                        .withContract(api.userSteps.balanceSteps().getDefaultContract(User.get(operator).getClientID()))
        );

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps()
                        .accountAmountChangedOnSum(account, amount.floatValue()), equalTo(true));
    }
}
