package ru.yandex.autotests.directapi.finance.accountmanagement.deposit;

import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.common.api45.Payment;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;

/**
 * User: mariabye
 * Date: 17.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_DEPOSIT)
@RunWith(Parameterized.class)
public class CurrencyTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    static String contractID = User.get(FinanceLogins.AGENCY_DEPOSIT_1).getExternalContractID();

    @Parameterized.Parameter
    public PaymentMap payment;

    @Parameterized.Parameter(value = 1)
    public String operatorLogin;

    @Parameterized.Parameter(value = 2)
    public String clientLogin;

    @Parameterized.Parameter(value = 3)
    public boolean isYMOAUTH;

    @Parameterized.Parameters()
    public static Collection payments() {
        Object[][] data = new Object[][]{
                {
                        new PaymentMap(api.type()).withContract(contractID),
                        FinanceLogins.AGENCY_DEPOSIT_1,
                        FinanceLogins.DEPOSIT_SC_1,
                        false},
//                {
//                        new PaymentMap(api.type()).withOrigin(PayMethod.OVERDRAFT),
//                        FinanceLogins.DEPOSIT_FREE_CLIENT,
//                        FinanceLogins.DEPOSIT_FREE_CLIENT,
//                        true}, до починки ЯД DIRECT-65926
//                {
//                        new PaymentMap(api.type()).withOrigin(PayMethod.YM),
//                        FinanceLogins.DEPOSIT_FREE_CLIENT,
//                        FinanceLogins.DEPOSIT_FREE_CLIENT,
//                        true
//                } отключено до починки выдачи платежных токенов на koga (TESTIRT-10089)
        };
        return Arrays.asList(data);
    }

    Money minPaymentAmount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount();

    @BeforeClass
    public static void initTest() {
        api.as(FinanceLogins.AGENCY_DEPOSIT_1);
        api.userSteps.clientFakeSteps().enableAgencyWallet(FinanceLogins.AGENCY_DEPOSIT_1);
        api.userSteps.addModeratedCampaign(FinanceLogins.DEPOSIT_SC_1);
        api.userSteps.financeSteps().verifySharedAccount(FinanceLogins.DEPOSIT_SC_1);
        api.as(FinanceLogins.DEPOSIT_FREE_CLIENT).userSteps.addModeratedCampaign(FinanceLogins.DEPOSIT_FREE_CLIENT);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(FinanceLogins.DEPOSIT_FREE_CLIENT);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5971")
    public void setCurrencyRub() {
        int accountID = api.as(operatorLogin).userSteps.financeSteps().getAccountID(clientLogin);
        payment.withAccountID(accountID);
        payment.withCurrency(Currency.RUB.toString());
        payment.withAmount(minPaymentAmount.addVAT().floatValue());

        Account account = api.userSteps.financeSteps().getAccount(((Payment) payment.getBean()).getAccountID());
        api.userSteps.financeSteps().deposit(isYMOAUTH, payment);


        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps().accountAmountChangedOnSum(
                        account, minPaymentAmount.floatValue()), equalTo(true));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5972")
    public void setCurrencyInvalid() {
        int accountID = api.as(operatorLogin).userSteps.financeSteps().getAccountID(clientLogin);
        payment.withAccountID(accountID);
        payment.withCurrency(Currency.UAH.toString());
        payment.withAmount(minPaymentAmount.floatValue());
        api.userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new PaymentMap[]{payment}, isYMOAUTH, new AxisError(245, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5973")
    public void setCurrencyEmpty() {
        int accountID = api.as(operatorLogin).userSteps.financeSteps().getAccountID(clientLogin);
        payment.withAccountID(accountID);
        payment.withCurrency("");
        payment.withAmount(minPaymentAmount.floatValue());
        api.userSteps
                .financeSteps().shouldGetErrorOnAMDeposit(
                new AxisError(245), isYMOAUTH, payment);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("5974")
    public void payAccountTwiceWithCurrency() {
        int accountID = api.as(operatorLogin).userSteps.financeSteps().getAccountID(clientLogin);
        payment.withAccountID(accountID);
        payment.withCurrency(Currency.RUB.toString());
        payment.withAmount(minPaymentAmount.addVAT().floatValue());
        Account account = api.userSteps.financeSteps().getAccount(((Payment) payment.getBean()).getAccountID());
        api.userSteps.financeSteps().deposit(isYMOAUTH, payment, payment);

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps().accountAmountChangedOnSum(
                        account, minPaymentAmount.add(minPaymentAmount.floatValue()).floatValue()), equalTo(true));
    }
}

