package ru.yandex.autotests.directapi.finance.accountmanagement.deposit;

import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayMethod;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_DEPOSIT)
@RunWith(Parameterized.class)
@Description("Проверка запрета оплаты общего счёта с помощью метода AccountManagement.Deposit в гривнах")
@Issue("https://st.yandex-team.ru/DIRECT-66515")
public class DepositCurrencyUAHNegativeTest {

    private static final String agencyLogin = Logins.AGENCY_UAH;
    private static final String subclientLogin = Logins.SUBCLIENT_UAH;
    private static final String clientLogin = FinanceLogins.DEPOSIT_BY_CARD_UAH;
    private static final String randomContractID = "12345"; // random number
    private static final float randomPaymentAmount = 1000.0f;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static int subclientAccountID;
    private static int clientAccountID;

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter
    public Currency currency;

    private LogSteps log = LogSteps.getLogger(this.getClass());

    @Parameterized.Parameters()
    public static Collection payments() {
        Object[][] data = new Object[][]{
                {null},
                {Currency.UAH},
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void initTest() {
        api.as(agencyLogin);
        api.userSteps.clientFakeSteps().enableAgencyWallet(agencyLogin);
        api.as(agencyLogin).userSteps.addModeratedCampaign(subclientLogin);
        api.userSteps.financeSteps().verifySharedAccount(subclientLogin);

        subclientAccountID = api.as(agencyLogin).userSteps.financeSteps().getAccountID(subclientLogin);

        api.as(clientLogin);
        api.userSteps.addModeratedCampaign(clientLogin);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(clientLogin);

        clientAccountID = api.as(clientLogin).userSteps.financeSteps().getAccountID(clientLogin);
    }

    @Test
    public void checkCreditLimit() {
        String curr = currency != null ? currency.toString() : null;
        log.info("Вызываем метод AccountManagement.Deposit для украинского субклиента. Currency " + curr);

        api.as(agencyLogin).userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new AxisError[]{new AxisError(245)},
                new PaymentMap(api.type())
                        .withContract(randomContractID)
                        .withAccountID(subclientAccountID)
                        .withCurrency(currency)
                        .withAmount(randomPaymentAmount));
    }

    @Test
    public void checkOverdraft() {
        String curr = currency != null ? currency.toString() : null;
        log.info("Вызываем метод AccountManagement.Deposit для украинского клиента. Currency " + curr);

        api.as(clientLogin).userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new AxisError[]{new AxisError(245)},
                new PaymentMap(api.type())
                        .withOrigin(PayMethod.OVERDRAFT)
                        .withAccountID(clientAccountID)
                        .withCurrency(currency)
                        .withAmount(randomPaymentAmount));
    }
}
