package ru.yandex.autotests.directapi.finance.accountmanagement.deposit;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.matchers.finance.AccountMatcher;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayMethod;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * User: mariabye
 * Date: 17.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_DEPOSIT)
@Tag(TagDictionary.TRUNK)
public class DepositOverdraftModeTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    static String login = FinanceLogins.FREE_CLIENT_OVERDRAFT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static float minPaymentAmount;

    @BeforeClass
    public static void getAccountIDs() {
        api.as(FinanceLogins.AGENCY_DEPOSIT_1);
        api.userSteps.clientFakeSteps().enableAgencyWallet(FinanceLogins.AGENCY_DEPOSIT_1);
        api.userSteps.createAccount(FinanceLogins.DEPOSIT_SC_1);
        api.as(FinanceLogins.INVOICE_FREE_CLIENT).userSteps.addModeratedCampaign(FinanceLogins.INVOICE_FREE_CLIENT);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(FinanceLogins.INVOICE_FREE_CLIENT);
        minPaymentAmount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().floatValue();
        api.as(login).userSteps.addModeratedCampaign(login);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(login);
        api.userSteps.balanceSteps().synchronizeWithBalance(
                ((Account) api.as(login).userSteps.financeSteps().getAccount(login)).getAccountID());
    }


    //@Test до починки ЯД DIRECT-65926
    @ru.yandex.qatools.allure.annotations.TestCaseId("6002")
    public void payAccountWithPaymentToken() {
        Account account = api.as(login).userSteps.financeSteps().getAccount(login);

        api.userSteps.financeSteps().deposit(true,
                new PaymentMap(api.type())
                        .withAccountID(account.getAccountID())
                        .withOrigin(PayMethod.OVERDRAFT)
                        .withAmount(Money.valueOf(minPaymentAmount, Currency.RUB).addVAT().floatValue())
                        .withCurrency(Currency.RUB)

        );

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps().accountAmountChangedOnSum(account, minPaymentAmount), equalTo(true));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6003")
    public void payAccountWithFinanceToken() {
        Account account = api.as(login).userSteps.financeSteps().getAccount(login);
        Float amount = account.getAmount();

        api.userSteps.financeSteps().deposit(
                new PaymentMap(api.type())
                        .withAccountID(account.getAccountID())
                        .withOrigin(PayMethod.OVERDRAFT)
                        .withAmount(Money.valueOf(minPaymentAmount, Currency.RUB).addVAT().floatValue())
                        .withCurrency(Currency.RUB)

        );


        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps().accountAmountChanged(account), equalTo(true));
        account = api.userSteps.financeSteps().getAccount(account.getAccountID());
        assertThat(account, AccountMatcher.hasAmount(equalTo(amount + minPaymentAmount)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6004")
    public void payAccountOverdraftNotAvailable() {
        float overdraftSum = api.userSteps.balanceSteps().getClientOverdraft(FinanceLogins.INVOICE_FREE_CLIENT);
        assumeThat("Клиенту разрешен овердрафт. Требуется отключить", overdraftSum, equalTo(0f));
        Account account = api.as(FinanceLogins.INVOICE_FREE_CLIENT).userSteps.financeSteps().getAccount(
                FinanceLogins.INVOICE_FREE_CLIENT
        );
        api.userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new PaymentMap[]{new PaymentMap(api.type())
                        .withAccountID(account.getAccountID())
                        .withOrigin(PayMethod.OVERDRAFT)
                        .withAmount(Money.valueOf(minPaymentAmount, Currency.RUB).addVAT().floatValue())
                        .withCurrency(Currency.RUB)},
                new AxisError(363, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6005")
    public void payAccountOverdraftAmountExceeded() {
        Account account = api.as(login).userSteps.financeSteps().getAccount(login);
        api.userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new PaymentMap[]{new PaymentMap(api.type())
                        .withAccountID(account.getAccountID())
                        .withOrigin(PayMethod.OVERDRAFT)
                        .withCurrency(Currency.RUB)
                        .withAmount(Money.valueOf(api.userSteps.balanceSteps().getClientOverdraft(login) + 1.0f,
                        Currency.YND_FIXED).convert(Currency.RUB).addVAT())},
                new AxisError(364, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6006")
    public void payAccountByAgencyNotAvailable() {
        Account account = api.as(login).userSteps.financeSteps().getAccount(login);
        api.as(FinanceLogins.AGENCY_DEPOSIT_1).userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new PaymentMap[]{new PaymentMap(api.type())
                        .withAccountID(account.getAccountID())
                        .withOrigin(PayMethod.OVERDRAFT)
                        .withAmount(Money.valueOf(minPaymentAmount, Currency.RUB).addVAT().floatValue())
                        .withCurrency(Currency.RUB)},
                new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6007")
    public void payAccountByManagerNotAvailable() {
        Account account = api.as(login).userSteps.financeSteps().getAccount(login);
        log.info("DIRECT-25758");
        api.as(FinanceLogins.MANAGER_LOGIN).userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new PaymentMap[]{new PaymentMap(api.type())
                        .withAccountID(account.getAccountID())
                        .withOrigin(PayMethod.OVERDRAFT)
                        .withAmount(Money.valueOf(minPaymentAmount, Currency.RUB).addVAT().floatValue())
                        .withCurrency(Currency.RUB)},
                new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6001")
    public void subClientNoRightToPay() {
        Account account = api.as(FinanceLogins.AGENCY_DEPOSIT_1)
                .userSteps.financeSteps().getAccount(FinanceLogins.DEPOSIT_SC_2);
        api.as(FinanceLogins.DEPOSIT_SC_2).userSteps.financeSteps().shouldGetErrorsOnAMDeposit(
                new PaymentMap[]{new PaymentMap(api.type())
                        .withAccountID(account.getAccountID())
                        .withOrigin(PayMethod.OVERDRAFT)
                        .withAmount(Money.valueOf(minPaymentAmount, Currency.RUB).addVAT().floatValue())
                        .withCurrency(Currency.RUB)},
                new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }
}
