package ru.yandex.autotests.directapi.finance.accountmanagement.depositebycard;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.AccountManagementRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;

/**
 * Created by semkagtn on 07.10.15.
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_DEPOSIT_BY_CARD)
@Description("Проверка минимального значения Amount для рублёвой валюты")
@Issue("https://st.yandex-team.ru/DIRECT-46802")
public class DepositByCardRubMinimumAmountTest {

    private static final String CLIENT = FinanceLogins.DEPOSIT_BY_CARD_RUB;
    private static final Currency CURRENCY = Currency.RUB;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @Rule
    public Trashman trasher = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static String customTransactionId;
    private static int accountId;

    @BeforeClass
    public static void createCampaign() {
        api.userSteps.addModeratedCampaign(CLIENT);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(CLIENT);
        accountId = api.as(CLIENT).userSteps.financeSteps().getAccountID(CLIENT);

        api.as(CLIENT);
        customTransactionId = randomAlphanumeric(AccountManagementRequestMap.CUSTOM_TRANSACTION_ID_LENGTH);
    }

    @Test
    @Description("Минимальнео значение Amount для рублёвой валюты в запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("6026")
    public void minAmountValueInRequest() {
        float amount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT().floatValue();

        api.userSteps.financeSteps().shouldGetErrorsOnAMDepositByCard("INVALID", customTransactionId, "20502A",
                new PaymentMap(api.type())
                        .withAmount(amount)
                        .withAccountID(accountId)
                        .withCurrency(Currency.RUB),
                new AxisError(368, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @Description("Значение Amount меньше минимального для рублёвой валюты в запросе")
    @ru.yandex.qatools.allure.annotations.TestCaseId("6027")
    public void lessThanMinAmountValueInRequest() {
        Money minSum = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount();

        api.userSteps.financeSteps().shouldGetErrorsOnAMDepositByCard("INVALID", customTransactionId, "20502A",
                new PaymentMap(api.type())
                        .withAmount(minSum.addVAT().getPrevious().floatValue())
                        .withAccountID(accountId)
                        .withCurrency(CURRENCY),
                new AxisError(354, AxisErrorDetails.LOW_CURRENCY_PAYMENT_AMOUNT_FOR_ACCOUNT,
                        minSum.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        MoneyCurrency.get(CURRENCY).getAPIAbbreviation(api.locale()), accountId));
    }
}
