package ru.yandex.autotests.directapi.finance.accountmanagement.get;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.CampaignsDayBudgetShowMode;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.Campaigns;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.enums.CampaignType;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.common.SpendMode;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasProperty;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;

@Aqua.Test
@RunWith(Parameterized.class)
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_GET_UPDATE)
public class AccountManagementGetAccountDailyBudgetSpendModeTest {
    private static final String LOGIN = FinanceLogins.ACCOUNT_DAILY_BUDGET_LOGIN4;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104).as(Logins.LOGIN_SUPER, LOGIN);
    private static long accountId;
    @Rule
    public Trashman trasher = new Trashman(api);
    @Parameterized.Parameter
    public String spendMode;
    @Parameterized.Parameter(1)
    public CampaignsDayBudgetShowMode databaseSpendMode;

    @BeforeClass
    public static void init() {
        // У клиента должна быть хотя бы одна кампания, кроме кошелька
        List<CampaignGetItem> campaigns = api.userSteps.campaignSteps().getCampaigns(LOGIN);
        if (campaigns.size() == 0) {
            api.userSteps.campaignSteps().addDefaultCampaign(CampaignType.TEXT);
        }

        api.userSteps.financeSteps().verifySharedAccount(LOGIN);
        accountId = api.userSteps.financeSteps().getAccountID(LOGIN);
    }

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> parameters() {
        Map<String, CampaignsDayBudgetShowMode> literalToDbEnum = Arrays.stream(CampaignsDayBudgetShowMode.values())
                .collect(Collectors.toMap(CampaignsDayBudgetShowMode::getLiteral, Function.identity()));

        return Arrays.stream(SpendMode.values())
                .sorted()
                .map(spendMode -> new Object[]{
                        spendMode.getValue(),
                        literalToDbEnum.get(spendMode.getValue().toLowerCase())})
                .collect(Collectors.toList());
    }

    @Test
    public void test() {
        api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(LOGIN)
                .campaignsSteps()
                .updateCampaigns(new CampaignsRecord()
                        .with(Campaigns.CAMPAIGNS.CID, accountId)
                        .with(Campaigns.CAMPAIGNS.DAY_BUDGET, BigDecimal.valueOf(500f))
                        .with(Campaigns.CAMPAIGNS.DAY_BUDGET_SHOW_MODE, databaseSpendMode));

        assertThat(
                "режим траты дневного бюджета из базы вернулся верно",
                api.userSteps.financeSteps().<Account>getAccount((int) accountId),
                hasProperty("accountDayBudget", hasProperty("spendMode", equalTo(spendMode))));
    }
}
