package ru.yandex.autotests.directapi.finance.accountmanagement.get;

import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.FixMethodOrder;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.MethodSorters;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.common.api45.AccountManagementResponse;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.objecthelpers.uiclient.data.types.Role;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;

/**
 * User: rockman
 * Date: 07.10.13
 */
@Aqua.Test
@RunWith(Parameterized.class)
@FixMethodOrder(MethodSorters.JVM)
@Tag(TagDictionary.TRUNK)
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_GET_UPDATE)
public class AgencyAndManagerAccountManagementGetTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameters
    public static Collection roleValues() {
        Object[][] data = new Object[][]{
                {FinanceLogins.AGENCY_TRANSFER_1, FinanceLogins.TRANSFER_CLIENT_1, FinanceLogins.TRANSFER_CLIENT_2,
                        FinanceLogins.CLIENT_WITHOUT_OS},
                {FinanceLogins.MANAGER_LOGIN, FinanceLogins.SERVICED_CLIENT1, FinanceLogins.SERVICED_CLIENT2,
                        FinanceLogins.TRANSFER_LOGIN},
        };
        return Arrays.asList(data);
    }

    public AgencyAndManagerAccountManagementGetTest(String agencyOrManagerLogin, String firstClientLogin,
            String secondClientLogin, String thirdClientWithoutES)
    {
        this.firstClientLogin = firstClientLogin;
        this.secondClientLogin = secondClientLogin;
        this.thirdClientWithoutES = thirdClientWithoutES;
        this.agencyOrManagerLogin = agencyOrManagerLogin;
        api.as(agencyOrManagerLogin);
        firstClientAccountId = api.userSteps.financeSteps().getAccountID(firstClientLogin);
        secondClientAccountId = api.userSteps.financeSteps().getAccountID(secondClientLogin);
    }

    private static int firstClientAccountId;
    private static int secondClientAccountId;

    private String firstClientLogin;
    private String secondClientLogin;
    private String thirdClientWithoutES;
    private String agencyOrManagerLogin;

    @BeforeClass
    public static void initTest() {
        api.as(Logins.SUPER_LOGIN);
        api.userSteps.addModeratedCampaign(FinanceLogins.SERVICED_CLIENT1);
        api.userSteps.financeSteps().verifySharedAccount(FinanceLogins.SERVICED_CLIENT1);
        api.userSteps.addModeratedCampaign(FinanceLogins.SERVICED_CLIENT2);
        api.userSteps.financeSteps().verifySharedAccount(FinanceLogins.SERVICED_CLIENT2);
        api.userSteps.clientFakeSteps().enableAgencyWallet(FinanceLogins.AGENCY_TRANSFER_1);
        api.as(FinanceLogins.AGENCY_TRANSFER_1).userSteps.financeSteps()
                .verifySharedAccount(FinanceLogins.TRANSFER_CLIENT_2);
        api.userSteps.clientFakeSteps().enableToCreateSelfCampaigns(FinanceLogins.TRANSFER_CLIENT_2);
        api.as(Logins.MANAGER_DEFAULT);
        api.userSteps.addDraftCampaign(FinanceLogins.TRANSFER_CLIENT_2);
        api.userSteps.clientFakeSteps().enableAgencyWallet(FinanceLogins.AGENCY_TRANSFER_1);
        api.as(FinanceLogins.AGENCY_TRANSFER_1).userSteps.createAccount(FinanceLogins.TRANSFER_CLIENT_1);
        api.userSteps.clientFakeSteps().enableToCreateSelfCampaigns(FinanceLogins.TRANSFER_CLIENT_1);
        api.as(Logins.MANAGER_DEFAULT);
        api.userSteps.campaignFakeSteps().resetWalletCreateDate(
                api.userSteps.addDraftCampaign(FinanceLogins.TRANSFER_CLIENT_1));
        api.as(FinanceLogins.AGENCY_TRANSFER_1).userSteps.financeSteps()
                .verifySharedAccount(FinanceLogins.TRANSFER_CLIENT_1);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(FinanceLogins.GET_UPDATE_CLIENT_1);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6038")
    public void getRequestByLogins() {
        String[] logins = new String[]{firstClientLogin, secondClientLogin};
        AccountManagementResponse response = api.userSteps.financeSteps().get(logins, null);
        assertThat("Количество возвращенных аккаунтов не соответствует ожидаемому",
                response.getAccounts().length, equalTo(2));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6039")
    public void getRequestByIds() {
        int[] ids = new int[]{firstClientAccountId, secondClientAccountId};
        AccountManagementResponse response = api.userSteps.financeSteps().get(null, ids);
        assertThat("Количество возвращенных аккаунтов не соответствует ожидаемому",
                response.getAccounts().length, equalTo(2));
        for (Account account : response.getAccounts()) {
            if (account.getAccountID() == firstClientAccountId) {
                assertThat(account.getLogin(), equalTo(firstClientLogin));
            } else if (account.getAccountID() == secondClientAccountId) {
                assertThat(account.getLogin(), equalTo(secondClientLogin));
            } else {
                throw new DirectAPIException("Полученный аккаун не соответствует заявленному списку." +
                        " Login: " + account.getLogin() + ", ID: " + account.getAccountID());
            }
        }
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6040")
    public void getErrorRequestByLoginAndId() {
        api.userSteps.financeSteps()
                .shouldGetErrorOnAccountManagementGet(new String[]{firstClientLogin}, new int[]{secondClientAccountId},
                        new AxisError(71, AxisErrorDetails.EITHER_LOGINS_EITHER_ACCOUNT_IDS_REQUIRED));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6041")
    public void getErrorWithLoginWithoutES() {
        api.userSteps.financeSteps().shouldGetErrorsOnAccountManagementGet(
                thirdClientWithoutES, new AxisError(515, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6042")
    public void getErrorRequestWithDigitsLogin() {
        api.userSteps.financeSteps().shouldGetErrorsOnAccountManagementGet(
                "123", new AxisError(259, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6043")
    public void getErrorRequestWithAnotherLogin() {
        if (User.get(agencyOrManagerLogin).getRole().equals(Role.AGENCY)) {
            api.userSteps.financeSteps().shouldGetErrorsOnAccountManagementGet(
                    FinanceLogins.GET_UPDATE_CLIENT_1, new AxisError(54, AxisErrorDetails.NOT_AN_AGENCY_CLIENT));
        } else {
            api.userSteps.financeSteps().shouldGetErrorsOnAccountManagementGet(
                    FinanceLogins.GET_UPDATE_CLIENT_1,
                    new AxisError(54, AxisErrorDetails.YOU_ARE_NOT_AUTHORIZED_TO_VIEW_THE_REQUESTED_INFORMATION));
        }
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6044")
    public void getErrorRequestWithAnotherAccountId() {
        int accountID = 9937144;
        api.userSteps.financeSteps().shouldGetErrorsOnAccountManagementGet(accountID,
                new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6045")
    public void getErrorRequestWithCampaignIDInAccountIDS() {
        long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(firstClientLogin);
        api.userSteps.financeSteps().shouldGetErrorsOnAccountManagementGet(campaignID,
                new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }
}
