package ru.yandex.autotests.directapi.finance.accountmanagement.invoice;

import java.util.Arrays;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.finance.FinanceSteps;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;

/**
 * User: mariabye
 * Date: 14.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_INVOICE)
@Tag(TagDictionary.TRUNK)
public class InvoiceAgencyAccountTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class);


    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    Money minInvoiceAmount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT();

    static String agencyLogin1 = FinanceLogins.AGENCY_INVOICE_1;
    static String agency1SubClient1Login = FinanceLogins.INVOICE_SC_1;
    static String agency1SubClient2Login = FinanceLogins.INVOICE_SC_2;
    static String agency1SubClientNoAccountLogin = FinanceLogins.INVOICE_SC_11;

    static String agencyLogin2 = FinanceLogins.AGENCY_INVOICE_2;
    static String agency2SubClientNoAccountLogin = FinanceLogins.INVOICE_SC_3;

    static String agencyFree = FinanceLogins.AGENCY_FREE_CLIENT_ACCOUNT;
    static String subClientFree = FinanceLogins.AGENCY_AND_FREE_CLIENT_YE;

    static int accountSubClient1 = 0;
    static int accountSubClient2 = 0;

    @BeforeClass
    public static void prepareAccounts() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(agencyFree);
        api.as(agencyFree).userSteps.createAccount(subClientFree);

        api.userSteps.clientFakeSteps().enableAgencyWallet(agencyLogin1);
        api.as(agencyLogin1).userSteps.createAccount(agency1SubClient1Login);
        api.as(agencyLogin1).userSteps.createAccount(agency1SubClient2Login);

        accountSubClient1 = api.as(agencyLogin1).userSteps.financeSteps().getAccountID(agency1SubClient1Login);
        accountSubClient2 = api.as(agencyLogin1).userSteps.financeSteps().getAccountID(agency1SubClient2Login);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6062")
    public void agencyPaySubClientAccount() {
        api.as(agencyLogin1);
        String requestID = api.userSteps.financeSteps()
                .getInvoiceRequestID(accountSubClient1, minInvoiceAmount.floatValue(), Currency.RUB);
        api.userSteps.balanceSteps().operator(User.get(agencyLogin1))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6066")
    public void couldNotPayClientAccountOfAnotherAgency() {
        api.as(agencyLogin2);
        api.userSteps.financeSteps()
                .shouldGetErrorsOnAMInvoice(accountSubClient1, minInvoiceAmount.floatValue(), Currency.RUB,
                        new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6067")
    public void agencyPaySubClientNoAccountCampaignID() {
        api.as(agencyLogin2);

        long campaignID = api.userSteps.addDraftCampaign(agency2SubClientNoAccountLogin);
        api.userSteps.financeSteps()
                .shouldGetErrorsOnAMInvoice(campaignID, minInvoiceAmount.floatValue(), Currency.RUB,
                        new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6068")
    public void agencyPaySubClientsWithAccounts() {
        api.as(agencyLogin1);

        String[] requestIDs = new String[]{
                api.userSteps.financeSteps()
                        .getInvoiceRequestID(accountSubClient1, minInvoiceAmount.floatValue(), Currency.RUB),
                api.userSteps.financeSteps()
                        .getInvoiceRequestID(accountSubClient2, minInvoiceAmount.floatValue(), Currency.RUB)
        };
        api.userSteps.balanceSteps().operator(User.get(agencyLogin1))
                .requestAmountShouldBe(requestIDs[0], equalTo(minInvoiceAmount.floatValue()));
        api.userSteps.balanceSteps().operator(User.get(agencyLogin1))
                .requestAmountShouldBe(requestIDs[1], equalTo(minInvoiceAmount.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6069")
    public void agencyPaySubClientWithAccountAndWithout() {
        api.as(agencyLogin1);
        long campaignID1 = api.userSteps.addDraftCampaign(agency1SubClientNoAccountLogin);
        int accountID = api.userSteps.financeSteps().getAccountID(agency1SubClient1Login);
        api.userSteps.financeSteps()
                .shouldGetErrorsOnAMInvoice(
                        new AxisError[]{new AxisError(54, AxisErrorDetails.EMPTY_STRING)},
                        new PaymentMap(api.type()).withAccountID(campaignID1)
                                .withAmount(minInvoiceAmount.floatValue())
                                .withCurrency(Currency.RUB),
                        new PaymentMap(api.type()).withAccountID(accountID)
                                .withAmount(minInvoiceAmount.floatValue())
                                .withCurrency(Currency.RUB)
                );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6070")
    public void agencyPaymentItemsMaxNumber() {
        api.as(agencyLogin1);
        PaymentMap paymentMap = new PaymentMap(api.type()).withAccountID(0)
                .withAmount(minInvoiceAmount.floatValue())
                .withCurrency(Currency.RUB);
        PaymentMap[] paymentMaps = new PaymentMap[FinanceSteps.PAYMENT_ITEMS_MAX_NUMBER];
        Arrays.fill(paymentMaps, paymentMap);
        AxisError[] errors = new AxisError[FinanceSteps.PAYMENT_ITEMS_MAX_NUMBER];
        Arrays.fill(errors, new AxisError(54, AxisErrorDetails.EMPTY_STRING));
        api.userSteps.financeSteps().shouldGetErrorsOnAMInvoice(errors, paymentMaps);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6071")
    public void agencyPaymentItemsLimit() {
        api.as(agencyLogin1);
        PaymentMap paymentMap = new PaymentMap(api.type()).withAccountID(0)
                .withAmount(0f)
                .withCurrency(Currency.RUB);
        PaymentMap[] paymentMaps = new PaymentMap[FinanceSteps.PAYMENT_ITEMS_MAX_NUMBER + 1];
        Arrays.fill(paymentMaps, paymentMap);
        api.userSteps.financeSteps()
                .shouldGetErrorOnAMInvoice(
                        new AxisError(241, AxisErrorDetails.THE_ARRAY_MUST_CONTAIN_NO_MORE_THAN_N_ELEMENTS,
                                "Payments", 50),
                        paymentMaps);

    }
}
