package ru.yandex.autotests.directapi.finance.accountmanagement.invoice;

import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45.Payment;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;

/**
 * User: mariabye
 * Date: 17.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_INVOICE)
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class InvoiceCurrencyTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    Money minInvoiceAmount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount();

    @Parameterized.Parameter
    public String clientLogin;

    @Parameterized.Parameter(value = 1)
    public String operatorLogin;

    @Parameterized.Parameters(name = "Client = {1}")
    public static Collection payments() {
        Object[][] data = new Object[][]{
                {
                        FinanceLogins.INVOICE_SC_1,
                        FinanceLogins.AGENCY_INVOICE_1},
                {
                        FinanceLogins.INVOICE_FREE_CLIENT,
                        FinanceLogins.INVOICE_FREE_CLIENT}
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(FinanceLogins.AGENCY_INVOICE_1);
        api.as(FinanceLogins.AGENCY_INVOICE_1).userSteps.createAccount(FinanceLogins.INVOICE_SC_1);
        api.as(FinanceLogins.INVOICE_FREE_CLIENT).userSteps.addModeratedCampaign(FinanceLogins.INVOICE_FREE_CLIENT);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(FinanceLogins.INVOICE_FREE_CLIENT);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6086")
    public void setCurrencyInvalid() {
        api.as(operatorLogin);
        int accountID = api.userSteps.financeSteps().getAccountID(clientLogin);
        PaymentMap payment = new PaymentMap(api.type())
                .withAccountID(accountID)
                .withCurrency(Currency.UAH)
                .withAmount(minInvoiceAmount.addVAT().floatValue());

        api.userSteps.financeSteps()
                .shouldGetErrorsOnAMInvoice(
                        new AxisError[]{new AxisError(245, AxisErrorDetails.EMPTY_STRING)}, payment);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6087")
    public void setCurrencyEmpty() {
        api.as(operatorLogin);
        int accountID = api.userSteps.financeSteps().getAccountID(clientLogin);
        PaymentMap payment = new PaymentMap(api.type())
                .withAccountID(accountID)
                .withCurrency("")
                .withAmount(minInvoiceAmount.addVAT().floatValue());
        api.userSteps
                .financeSteps().shouldGetErrorOnAMInvoice(new AxisError(245), payment);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6082")
    public void twoCurrencyInvoiceForAccount() {
        api.as(operatorLogin);
        int accountID = api.userSteps.financeSteps().getAccountID(clientLogin);
        PaymentMap payment = new PaymentMap(api.type())
                .withAccountID(accountID)
                .withCurrency(Currency.RUB)
                .withAmount(minInvoiceAmount.addVAT().floatValue());
        String[] requestIDs = api.userSteps.financeSteps().getInvoiceRequestIDs(payment, payment);
        api.userSteps.balanceSteps().operator(User.get(operatorLogin))
                .requestAmountShouldBe(requestIDs[0],
                        equalTo(((Payment) payment.getBean()).getAmount()));

        api.userSteps.balanceSteps().operator(User.get(operatorLogin))
                .requestAmountShouldBe(requestIDs[1],
                        equalTo(((Payment) payment.getBean()).getAmount()));
    }
}

