package ru.yandex.autotests.directapi.finance.accountmanagement.invoice;

import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_INVOICE)
@RunWith(Parameterized.class)
@Description("Проверка запрета оплаты общего счёта с помощью метода AccountManagement.Invoice в гривнах")
@Issue("https://st.yandex-team.ru/DIRECT-66515")
public class InvoiceCurrencyUAHNegativeTest {

    private static final String agencyLogin = Logins.AGENCY_UAH;
    private static final String subclientLogin = Logins.SUBCLIENT_UAH;
    private static final String clientLogin = FinanceLogins.DEPOSIT_BY_CARD_UAH;
    private static final String randomContractID = "12345";
    private static final float randomPaymentAmount = 1000.0f;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter
    public Currency currency;

    private LogSteps log = LogSteps.getLogger(this.getClass());

    @Parameterized.Parameters()
    public static Collection payments() {
        Object[][] data = new Object[][]{
                {null},
                {Currency.UAH},
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void initTest() {
        api.as(agencyLogin);
        api.userSteps.clientFakeSteps().enableAgencyWallet(agencyLogin);
        api.as(agencyLogin).userSteps.addModeratedCampaign(subclientLogin);
        api.userSteps.financeSteps().verifySharedAccount(subclientLogin);

        api.as(clientLogin);
        api.userSteps.addModeratedCampaign(clientLogin);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(clientLogin);
    }

    @Test
    public void checkAgencyInvoice() {
        String curr = currency != null ? currency.toString() : null;
        log.info("Вызываем метод AccountManagement.Invoice для украинского субклиента. Currency " + curr);

        int accountID = api.as(agencyLogin).userSteps.financeSteps().getAccountID(subclientLogin);

        api.userSteps.financeSteps().shouldGetErrorsOnAMInvoice(
                new AxisError[]{new AxisError(245)},
                new PaymentMap(api.type())
                        .withContract(randomContractID)
                        .withAccountID(accountID)
                        .withCurrency(currency)
                        .withAmount(randomPaymentAmount));
    }

    @Test
    public void checkClientInvoice() {
        String curr = currency != null ? currency.toString() : null;
        log.info("Вызываем метод AccountManagement.Invoice для украинского клиента. Currency " + curr);

        int accountID = api.as(clientLogin).userSteps.financeSteps().getAccountID(clientLogin);

        api.userSteps.financeSteps().shouldGetErrorsOnAMInvoice(
                new AxisError[]{new AxisError(245)},
                new PaymentMap(api.type())
                        .withContract(randomContractID)
                        .withAccountID(accountID)
                        .withCurrency(currency)
                        .withAmount(randomPaymentAmount));
    }

}
