package ru.yandex.autotests.directapi.finance.accountmanagement.invoice;


import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45.Payment;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;

/**
 * User: mariabye
 * Date: 14.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_INVOICE)
@RunWith(Parameterized.class)
public class InvoiceManagerAmountTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    static Money servicedMinPaymentValue = MoneyCurrency.get(Currency.RUB).getMinServicedInvoiceAmount().addVAT();
    static Money minInvoiceAmount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT();


    @Parameterized.Parameter()
    public String operatorLogin;

    @Parameterized.Parameters(name = "Login = {0}")
    public static Collection payments() {
        Object[][] data = new Object[][]{
                {FinanceLogins.MANAGER_LOGIN},
                {FinanceLogins.INVOICE_FREE_CLIENT}
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void initTest() {
        api.as(FinanceLogins.INVOICE_FREE_CLIENT).userSteps.addModeratedCampaign(FinanceLogins.INVOICE_FREE_CLIENT);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(FinanceLogins.INVOICE_FREE_CLIENT);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6092")
    public void lowFirstPayment() {
        api.as(operatorLogin);
        PaymentMap payment =
                new PaymentMap(api.type())
                        .withAccountID(
                                api.userSteps.financeSteps().getAccountID(FinanceLogins.INVOICE_FREE_CLIENT))
                        .withCurrency(Currency.RUB)
                        .withAmount(servicedMinPaymentValue.getPrevious().floatValue());

        String requestID = api.userSteps.financeSteps().getInvoiceRequestIDs(payment)[0];
        api.userSteps.balanceSteps().operator(User.get(FinanceLogins.INVOICE_FREE_CLIENT))
                .requestAmountShouldBe(requestID, equalTo(servicedMinPaymentValue.getPrevious().floatValue()));
    }

    @Test
    //https://jira.yandex-team.ru/browse/DIRECT-25530
    @ru.yandex.qatools.allure.annotations.TestCaseId("6090")
    public void lowAmountWithCurrency() {
        api.as(operatorLogin);
        PaymentMap payment =
                new PaymentMap(api.type())
                        .withAccountID(
                                api.userSteps.financeSteps().getAccountID(FinanceLogins.INVOICE_FREE_CLIENT))
                        .withCurrency(Currency.RUB)
                        .withAmount(minInvoiceAmount.getPrevious().floatValue());

        api.userSteps.financeSteps().shouldGetErrorsOnAMInvoice(
                new AxisError[]{new AxisError(354, AxisErrorDetails.LOW_CURRENCY_PAYMENT_AMOUNT_FOR_ACCOUNT,
                        MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount()
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        MoneyCurrency.get(Currency.RUB).getApiDescription(),
                        ((Payment) payment.getBean()).getAccountID())},
                payment);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6093")
    public void minimalAmountWithCurrency() {
        api.as(operatorLogin);
        PaymentMap payment =
                new PaymentMap(api.type())
                        .withAccountID(
                                api.userSteps.financeSteps().getAccountID(FinanceLogins.INVOICE_FREE_CLIENT))
                        .withCurrency(Currency.RUB)
                        .withAmount(minInvoiceAmount.floatValue());

        String requestID = api.userSteps.financeSteps().getInvoiceRequestIDs(payment)[0];
        api.userSteps.balanceSteps().operator(User.get(FinanceLogins.INVOICE_FREE_CLIENT))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.floatValue()));
    }
}
