package ru.yandex.autotests.directapi.finance.accountmanagement.transfer;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.model.ShardNumbers;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PaymentMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;

/**
 * User: pavryabov
 * Date: 22.01.14
 * <p>
 * https://jira.yandex-team.ru/browse/TESTIRT-1429
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_TRANSFER)
@RunWith(Parameterized.class)
public class AccountManagementTransferShardingPositiveTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public String operator;

    @Parameterized.Parameter(value = 1)
    public String clientFrom;

    @Parameterized.Parameter(value = 2)
    public String clientTo;

    @Parameterized.Parameters(name = "Operator = {0}, ClientFrom = {1}, ClientTo = {2}")
    public static Collection strategies() {
        Object[][] data = new Object[][]{
                {Logins.AGENCY_SHARD_2, Logins.ACCOUNT_SUBCLIENT_SHARD, Logins.ACCOUNT_SUBCLIENT_SHARD1}
        };
        return Arrays.asList(data);
    }

    public Account accountFrom;
    public Account accountTo;
    public Money amount;
    public Currency currency = null;

    @BeforeClass
    public static void reshard() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(Logins.AGENCY_SHARD_2);
        api.as(Logins.AGENCY_SHARD_2).userSteps.createAccount(Logins.ACCOUNT_SUBCLIENT_SHARD);
        api.userSteps.createAccount(Logins.ACCOUNT_SUBCLIENT_SHARD1);
        api.userSteps.clientFakeSteps().reshardUserWithLock(Logins.AGENCY_SHARD_2, ShardNumbers.EXTRA_SHARD);
        api.userSteps.clientFakeSteps().reshardUserWithLock(Logins.ACCOUNT_SUBCLIENT_SHARD, ShardNumbers.EXTRA_SHARD);
        api.userSteps.clientFakeSteps()
                .reshardUserWithLock(Logins.ACCOUNT_SUBCLIENT_SHARD1, ShardNumbers.DEFAULT_SHARD);
    }

    @Before
    public void initTest() {
        accountFrom = api.as(operator).userSteps.financeSteps().getAccount(clientFrom);
        accountTo = api.userSteps.financeSteps().getAccount(clientTo);
        api.userSteps.balanceSteps().synchronizeWithBalance(accountFrom.getAccountID());
        api.userSteps.balanceSteps().synchronizeWithBalance(accountTo.getAccountID());
        currency = Currency.getFor(accountFrom.getCurrency());
        amount = MoneyCurrency.get(currency).getMinTransferAmount();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6097")
    public void accountManagementTransferTest() {
        api.as(operator).userSteps.financeSteps().deposit(
                new PaymentMap(api.type())
                        .withAccountID(accountFrom.getAccountID())
                        .withCurrency(currency)
                        .withAmount(amount.addVAT().floatValue() * 2)
                        .withContract(api.userSteps.balanceSteps().getDefaultContract(User.get(operator).getClientID()))
        );

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps().accountAmountChanged(accountFrom), equalTo(true));
        accountFrom = api.userSteps.financeSteps().getAccount(clientFrom);
        accountTo = api.userSteps.financeSteps().getAccount(clientTo);
        api.userSteps.financeSteps()
                .transfer(accountFrom.getAccountID(), accountTo.getAccountID(), amount.floatValue(),
                        accountFrom.getCurrency());

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps()
                        .accountAmountChangedOnSum(accountFrom, -amount.floatValue()), equalTo(true));

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps()
                        .accountAmountChangedOnSum(accountTo, amount.floatValue()), equalTo(true));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6098")
    public void accountManagementReverseTransferTest() {
        api.as(operator).userSteps.financeSteps().deposit(
                new PaymentMap(api.type())
                        .withAccountID(accountTo.getAccountID())
                        .withCurrency(currency)
                        .withAmount(amount.addVAT().floatValue() * 2)
                        .withContract(api.userSteps.balanceSteps().getDefaultContract(User.get(operator).getClientID()))
        );

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps().accountAmountChanged(accountTo), equalTo(true));
        accountTo = api.userSteps.financeSteps().getAccount(clientTo);
        accountFrom = api.userSteps.financeSteps().getAccount(clientFrom);
        api.userSteps.financeSteps()
                .transfer(accountTo.getAccountID(), accountFrom.getAccountID(), amount.floatValue(),
                        accountFrom.getCurrency());

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps()
                        .accountAmountChangedOnSum(accountTo, -amount.floatValue()), equalTo(true));

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps()
                        .accountAmountChangedOnSum(accountFrom, amount.floatValue()), equalTo(true));
    }
}
