package ru.yandex.autotests.directapi.finance.accountmanagement.transfer;

import com.google.common.primitives.Ints;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.CreateNewSubclientResponse;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.TransferMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * User: pavryabov
 * Date: 18.10.13
 * <p>
 * https://jira.yandex-team.ru/browse/TESTIRT-568
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_TRANSFER)
public class TransferMoneyFailTest {

    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static final String MANAGER = FinanceLogins.MANAGER_LOGIN;
    private static final String AGENCY_LOGIN = FinanceLogins.AGENCY_TMONEY;
    private static final String ACCOUNT_CLIENT_LOGIN1 = FinanceLogins.TMONEY_CLIENT2;
    private static final String ACCOUNT_CLIENT_LOGIN2 = FinanceLogins.TMONEY_CLIENT9;
    private static final String NO_CURRENCY_AGENCY_LOGIN = FinanceLogins.AGENCY_TRANSFER_1;
    private static final String NO_CURRENCY_CLIENT_LOGIN = FinanceLogins.TRANSFER_CLIENT_1;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(AGENCY_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static float sumForCreate = 10000f;
    private static float sumForTransfer = 300f;

    private static int accountIDClient1;
    private static int accountIDClient2;
    private static int accountIDNoCurrencyAgency;

    @BeforeClass
    public static void getAccountId() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(NO_CURRENCY_AGENCY_LOGIN);
        api.as(NO_CURRENCY_AGENCY_LOGIN);
        api.userSteps.clientFakeSteps().enableToCreateSelfCampaigns(NO_CURRENCY_CLIENT_LOGIN);
        api.userSteps.createAccount(NO_CURRENCY_CLIENT_LOGIN);
        api.as(Logins.MANAGER_DEFAULT);
        Long campaignID = api.userSteps.addDraftCampaign(NO_CURRENCY_CLIENT_LOGIN);
        api.userSteps.campaignFakeSteps().resetWalletCreateDate(campaignID);
        api.as(NO_CURRENCY_AGENCY_LOGIN);
        api.userSteps.financeSteps().verifySharedAccount(NO_CURRENCY_CLIENT_LOGIN);
        accountIDNoCurrencyAgency = api.userSteps.financeSteps().getAccountID(NO_CURRENCY_CLIENT_LOGIN);

        api.userSteps.clientFakeSteps().enableAgencyWallet(AGENCY_LOGIN);
        api.as(AGENCY_LOGIN);
        api.userSteps.addAccountWithMoney(ACCOUNT_CLIENT_LOGIN1, sumForCreate, Currency.RUB);
        api.userSteps.createAccount(ACCOUNT_CLIENT_LOGIN2);
        accountIDClient1 = api.userSteps.financeSteps().getAccountID(ACCOUNT_CLIENT_LOGIN1);
        accountIDClient2 = api.userSteps.financeSteps().getAccountID(ACCOUNT_CLIENT_LOGIN2);
        api.userSteps.balanceSteps().synchronizeWithBalance(accountIDClient1);
        api.userSteps.balanceSteps().synchronizeWithBalance(accountIDClient2);
    }

    public TransferMoneyFailTest() {
        api.as(AGENCY_LOGIN);
    }

    //Невалидные запросы

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6101")
    public void expectErrorTransferCampaignIDInFrom() {
        long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(ACCOUNT_CLIENT_LOGIN1);
        api.userSteps.financeSteps().shouldGetMultipleErrorOnTransfer(
                Ints.checkedCast(campaignID), accountIDClient2, sumForTransfer, null,
                new AxisError(54, AxisErrorDetails.ACCOUNT, campaignID));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6102")
    public void expectErrorTransferCampaignIDInTo() {
        long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(ACCOUNT_CLIENT_LOGIN1);
        api.userSteps.financeSteps().shouldGetMultipleErrorOnTransfer(
                accountIDClient2, Ints.checkedCast(campaignID), sumForTransfer, null,
                new AxisError(54, AxisErrorDetails.ACCOUNT, campaignID));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6103")
    public void expectErrorClientCallTransfer() {
        api.as(Logins.CLIENT_FREE_YE_DEFAULT);
        api.userSteps.financeSteps().shouldGetErrorOnTransfer(
                accountIDClient1, accountIDClient2, sumForTransfer, null,
                new AxisError(71, AxisErrorDetails.FIELD_ACTION_FOR_CLIENT));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6104")
    public void expectErrorTransferWIthMoreThanOneTransfer() {
        api.userSteps.financeSteps().shouldGetErrorOnTransfer(
                new AxisError(241, AxisErrorDetails.THE_TRANSFERS_ARRAY_MUST_CONTAIN_NO_MORE_THAN_1_ELEMENTS),
                new TransferMap(api.type())
                        .withFromAccountID(accountIDClient1)
                        .withToAccountID(accountIDClient2)
                        .withAmount(sumForTransfer),
                new TransferMap(api.type())
                        .withFromAccountID(accountIDClient1)
                        .withToAccountID(accountIDClient2)
                        .withAmount(sumForTransfer)
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6105")
    public void expectErrorTransferWithZeroAmountValue() {
        api.userSteps.financeSteps().shouldGetErrorOnTransfer(
                accountIDClient1, accountIDClient2, 0, null,
                new AxisError(71, AxisErrorDetails.TRANSFERS_AMOUNT_SHOULDBE_MORE_THAN_ZERO));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6106")
    public void expectErrorTransferWithNegativeAmountValue() {
        api.userSteps.financeSteps().shouldGetErrorOnTransfer(
                accountIDClient1, accountIDClient2, -1, null,
                new AxisError(71, AxisErrorDetails.TRANSFERS_AMOUNT_SHOULDBE_MORE_THAN_ZERO));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6107")
    public void expectErrorTransferWithInvalidCurrency() {
        api.userSteps.financeSteps().shouldGetMultipleErrorOnTransfer(
                accountIDClient1, accountIDClient2, sumForTransfer, Currency.USD.toString(),
                new AxisError(245, AxisErrorDetails.ACCOUNT, accountIDClient1),
                new AxisError(245, AxisErrorDetails.ACCOUNT, accountIDClient2));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6109")
    public void expectErrorTransferAmountLessThanMinRUB() {
        float amount = MoneyCurrency.get(Currency.RUB).getMinTransferAmount().getPrevious().floatValue();
        api.userSteps.financeSteps().shouldGetMultipleErrorOnTransfer(
                accountIDClient1, accountIDClient2, amount, Currency.RUB.toString(),
                new AxisError(353, AxisErrorDetails.LOW_FUND_TRANSFERS_FROM,
                        MoneyCurrency.get(Currency.RUB).getMinTransferAmount()
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        MoneyCurrency.get(Currency.RUB).getApiDescription())
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6110")
    public void expectErrorTransferSameAccountInFromAndInTo() {
        api.userSteps.financeSteps().shouldGetMultipleErrorOnTransfer(
                accountIDClient1, accountIDClient1, sumForTransfer, null,
                new AxisError(353, AxisErrorDetails.ACCOUNTS_MUST_BE_DIFFERENT));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6111")
    public void expectErrorTransferNoCampaignsInWallet() {
        CreateNewSubclientResponse createNewSubclientResponse =
                api.userSteps.clientSteps().createNewAgencySubClient("subClient", AGENCY_LOGIN,
                        Currency.RUB);
        String subclient = createNewSubclientResponse.getLogin();
        long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(subclient);
        int accountID = api.userSteps.financeSteps().enableAndGetSharedAccount(subclient);
        api.userSteps.campaignSteps().campaignsDelete(subclient, campaignID);
        api.userSteps.financeSteps().shouldGetMultipleErrorOnTransfer(
                accountIDClient1, accountID, sumForTransfer, null,
                new AxisError(515, AxisErrorDetails.OPERATIONS_WITH_SHARED_ACCOUNT_DISABLED, accountID));
    }

    //Проверка прав под агентством

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6112")
    public void expectErrorTransferAnotherAgencyAccountInFrom() {
        log.info("Перевод между кошельком агентства и кошельком чужого клиента");
        api.userSteps.financeSteps().shouldGetMultipleErrorOnTransfer(
                accountIDNoCurrencyAgency, accountIDClient2, sumForTransfer, null,
                new AxisError(54, AxisErrorDetails.ACCOUNT, accountIDNoCurrencyAgency));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6113")
    public void expectErrorTransferAnotherAgencyAccountInTo() {
        log.info("Перевод между кошельком агентства и кошельком чужого клиента");
        api.userSteps.financeSteps().shouldGetMultipleErrorOnTransfer(
                accountIDClient1, accountIDNoCurrencyAgency, sumForTransfer, null,
                new AxisError(54, AxisErrorDetails.ACCOUNT, accountIDNoCurrencyAgency));
    }

    //Вызов под менеджером

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6116")
    public void expectErrorManagerTransferBetweenTwoAgencyClients() {
        log.info("Перевод под менеджером между кошельками двух клиентов");
        api.as(MANAGER);
        api.userSteps.financeSteps().shouldGetMultipleErrorOnTransfer(
                accountIDClient1, accountIDClient2, sumForTransfer, null,
                new AxisError(54, AxisErrorDetails.TRANSFER_FROM_ACCOUNT_ONLY_FOR_AGENCY));
    }
}
