package ru.yandex.autotests.directapi.finance.accountmanagement.update;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.CampOptions;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampOptionsRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.campaigns.EmailNotificationInfoMap;
import ru.yandex.autotests.directapi.model.campaigns.SmsNotificationInfoMap;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.AccountMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;

@Aqua.Test
@RunWith(Parameterized.class)
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_GET_UPDATE)
public class AccountManagementUpdateAccountDailyBudgetChangeNotificationSettingsRightsNegativeTest {
    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104);

    private static final String RESET_VALUE = ""; // в базе
    private static final String SET_VALUE = Value.YES; // в API

    @Rule
    public Trashman trasher = new Trashman(api);

    public static final class TestCase {
        private final String description;
        private final String operatorLogin;
        private final String clientLogin;

        private TestCase(String description, String operatorLogin, String clientLogin) {
            this.description = description;
            this.operatorLogin = operatorLogin;
            this.clientLogin = clientLogin;
        }

        @Override
        public String toString() {
            return description;
        }
    }

    private final TestCase testCase;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> parameters() {
        List<TestCase> parameters = Arrays.asList(
                new TestCase("представитель агентства не может поменять уведомления про дневной бюджет на общий счёт самоходному клиенту",
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_AGENCY_REP1_LOGIN,
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_REGULAR_CLIENT_LOGIN),
                new TestCase(
                        "менеджер не может поменять уведомления про дневной бюджет на общий счёт самоходному клиенту",
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_AGENCY_MANAGER_LOGIN,
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_REGULAR_CLIENT_LOGIN),
                new TestCase("агентский клиент без права на редактирование кампаний не может поменять уведомления про дневной бюджет на общий счёт себе",
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_AGENCY_CLIENT_LOGIN3,
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_AGENCY_CLIENT_LOGIN3),
                new TestCase("представитель агентства не может поменять уведомления про дневной бюджет на общий счёт не своему клиенту",
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_AGENCY_REP2_LOGIN,
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_AGENCY_CLIENT_LOGIN1),
                new TestCase("менеджер не может поменять уведомления про дневной бюджет на общий счёт клиенту не своего агентства",
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_IRRELEVANT_MANAGER_LOGIN,
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_AGENCY_CLIENT_LOGIN1),
                new TestCase("менеджер не может поменять уведомления про дневной бюджет на общий счёт не своему клиенту",
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_IRRELEVANT_MANAGER_LOGIN,
                        FinanceLogins.ACCOUNT_DAILY_BUDGET_REGULAR_CLIENT_LOGIN)
        );

        return parameters.stream()
                .map(testCase -> new Object[]{testCase})
                .collect(Collectors.toList());
    }

    public AccountManagementUpdateAccountDailyBudgetChangeNotificationSettingsRightsNegativeTest(TestCase testCase) {
        this.testCase = testCase;
    }

    private long accountId;

    @Before
    public void setUp() {
        api.as(Logins.LOGIN_SUPER, testCase.operatorLogin);

        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps()
                .useShardForLogin(testCase.clientLogin);

        Long clientId = jooqDbSteps.usersSteps().getUser(testCase.clientLogin).getClientid();
        accountId = jooqDbSteps.campaignsSteps().getWallet(clientId).getCid();

        jooqDbSteps.campaignsSteps()
                .updateCampOptions(new CampOptionsRecord()
                        .with(CampOptions.CAMP_OPTIONS.CID, accountId)
                        .with(CampOptions.CAMP_OPTIONS.EMAIL_NOTIFICATIONS, RESET_VALUE)
                        .with(CampOptions.CAMP_OPTIONS.SMS_FLAGS, RESET_VALUE));
    }

    @Test
    public void testCannotChangeDailyBudgetEmailNotificationSettings() {
        api.userSteps.financeSteps().shouldGetErrorOnUpdateAccount(
                new AxisError(54, AxisErrorDetails.NO_RIGHTS_TO_SHARED_ACCOUNT, accountId),
                new AccountMap(api.type())
                        .withAccountID((int) accountId)
                        .withEmailNotification(new EmailNotificationInfoMap(api.type()).on()
                                .withPausedByDayBudget(SET_VALUE)));
    }

    @Test
    public void testCannotChangeDailyBudgetSmsNotificationSettings() {
        api.userSteps.financeSteps().shouldGetErrorOnUpdateAccount(
                new AxisError(54, AxisErrorDetails.NO_RIGHTS_TO_SHARED_ACCOUNT, accountId),
                new AccountMap(api.type())
                        .withAccountID((int) accountId)
                        .withSmsNotification(new SmsNotificationInfoMap(api.type()).on()
                                .withPausedByDayBudgetSms(SET_VALUE)));
    }
}
