package ru.yandex.autotests.directapi.finance.accountmanagement.update;

import java.util.Arrays;
import java.util.Collection;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.common.api45.AccountManagementResponse;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.common.SpendMode;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.AccountDayBudgetInfoMap;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.AccountMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;

import static org.hamcrest.Matchers.arrayWithSize;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

@Aqua.Test
@RunWith(Parameterized.class)
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_GET_UPDATE)
public class AccountManagementUpdateAccountDailyBudgetSpendModeTest {
    private static final String LOGIN = FinanceLogins.ACCOUNT_DAILY_BUDGET_LOGIN8;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104).as(Logins.LOGIN_SUPER, LOGIN);

    private static final float DEFAULT_AMOUNT = 1000f;
    private static long accountId;

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter
    public SpendMode initialSpendMode;

    @Parameterized.Parameter(1)
    public SpendMode spendMode;

    @BeforeClass
    public static void initOnce() {
        api.userSteps.financeSteps().verifySharedAccount(LOGIN);
        accountId = api.userSteps.financeSteps().getAccountID(LOGIN);
    }

    @Parameterized.Parameters(name = "initial spend mode {0}, desired spend mode {1}")
    public static Collection<Object[]> parameters() {
        SpendMode[] spendModeValues = SpendMode.values();
        return Arrays.stream(spendModeValues)
                .sorted()
                .map(spendMode -> new Object[]{
                        Arrays.stream(spendModeValues).filter(e -> !e.equals(spendMode))
                                .collect(Collectors.toList()).get(0),
                        spendMode})
                .collect(Collectors.toList());
    }

    @Before
    public void prepareInitilaDailyBudgetSpendMode() {
        AccountMap account = new AccountMap(api.type())
                .withAccountID((int) accountId)
                .withAccountDayBudget(new AccountDayBudgetInfoMap(api.type())
                        .withSpendMode(initialSpendMode.getValue())
                        .withAmount(DEFAULT_AMOUNT));

        AccountManagementResponse response = api.userSteps.financeSteps().update(account);

        assumeThat("В ответе отсутствуют ошибки",
                (response.getActionsResult())[0].getErrors(), nullValue());

        assumeThat(
                "начальный режим траты дневного бюджета корректен",
                api.userSteps.getDirectJooqDbSteps()
                        .useShardForLogin(LOGIN)
                        .campaignsSteps()
                        .getCampaignById(accountId)
                        .getDayBudgetShowMode(),
                hasProperty("literal", equalTo(initialSpendMode.getValue().toLowerCase())));
    }

    @Test
    public void dailyBudgetSpendModeChanged() {
        AccountMap account = new AccountMap(api.type())
                .withAccountID((int) accountId)
                .withAccountDayBudget(new AccountDayBudgetInfoMap(api.type())
                        .withSpendMode(spendMode.getValue())
                        .withAmount(DEFAULT_AMOUNT));

        AccountManagementResponse response = api.userSteps.financeSteps().update(account);

        assumeThat("В ответе отсутствуют ошибки",
                (response.getActionsResult())[0].getErrors(), nullValue());

        assertThat(
                "режим траты дневного бюджета сохранился верно",
                api.userSteps.getDirectJooqDbSteps()
                        .useShardForLogin(LOGIN)
                        .campaignsSteps()
                        .getCampaignById(accountId)
                        .getDayBudgetShowMode(),
                hasProperty("literal", equalTo(spendMode.getValue().toLowerCase())));
    }

}
