package ru.yandex.autotests.directapi.finance.accountmanagement.update;

import java.util.Arrays;
import java.util.Collection;
import java.util.stream.Collectors;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.darkside.Logins;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.campaigns.EmailNotificationInfoMap;
import ru.yandex.autotests.directapi.model.campaigns.SmsNotificationInfoMap;
import ru.yandex.autotests.directapi.model.common.SpendMode;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.AccountDayBudgetInfoMap;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.AccountMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Features;

@Aqua.Test
@RunWith(Parameterized.class)
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_GET_UPDATE)
public class AccountManagementUpdateAccountDailyBudgetValidationTest {
    private static final String LOGIN = FinanceLogins.ACCOUNT_DAILY_BUDGET_LOGIN9;

    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104).as(Logins.LOGIN_SUPER, LOGIN);

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final String LOCALE = DirectTestRunProperties.getInstance().getDirectAPILocale();

    private static int accountId;

    public static class TestCase {
        private final String name;
        private final AccountMap account;
        private final AxisError expectedError;

        private TestCase(String name, AccountMap account, AxisError expectedError) {
            this.name = name;
            this.account = account;
            this.expectedError = expectedError;
        }

        @Override
        public String toString() {
            return name;
        }
    }

    @Parameterized.Parameter
    public TestCase testCase;

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> parameters() {
        String invalidValue = "InvalidValue";

        TestCase[] testCases = new TestCase[] {
                new TestCase("слишком маленький amount: 1",
                        new AccountMap(api.type())
                                .withAccountDayBudget(new AccountDayBudgetInfoMap(api.type())
                                        .withAmount(1f)
                                        .withSpendMode(SpendMode.DEFAULT.getValue())),
                        new AxisError(5005, AxisErrorDetails.DAILY_BUDGET_TOO_LOW,
                                "1\u00A0000.00", MoneyCurrency.get(Currency.RUB).getAPIAbbreviation(LOCALE))),
                new TestCase("слишком маленький amount: 999",
                        new AccountMap(api.type())
                                .withAccountDayBudget(new AccountDayBudgetInfoMap(api.type())
                                        .withAmount(999f)
                                        .withSpendMode(SpendMode.DEFAULT.getValue())),
                        new AxisError(5005, AxisErrorDetails.DAILY_BUDGET_TOO_LOW,
                                "1\u00A0000.00", MoneyCurrency.get(Currency.RUB).getAPIAbbreviation(LOCALE))),
                new TestCase("слишком большой amount",
                        new AccountMap(api.type())
                                .withAccountDayBudget(new AccountDayBudgetInfoMap(api.type())
                                        .withAmount(1.0e10f)
                                        .withSpendMode(SpendMode.DEFAULT.getValue())),
                        new AxisError(5005, AxisErrorDetails.DAILY_BUDGET_TOO_HIGH,
                                "1\u00A0000\u00A0000\u00A0000.00", MoneyCurrency.get(Currency.RUB).getAPIAbbreviation(LOCALE))),
                new TestCase("невозможный spendMode",
                        new AccountMap(api.type())
                                .withAccountDayBudget(new AccountDayBudgetInfoMap(api.type())
                                        .withAmount(1000f)
                                        .withSpendMode(invalidValue)),
                        new AxisError(71, AxisErrorDetails.FIELD_MUST_CONTAIN_ENUM_VALUE,
                                "SpendMode",
                                String.join(", ", SpendMode.DEFAULT.getValue(), SpendMode.STRETCHED.getValue()))),
                new TestCase("невозможный emailNotificationInfo.pausedByDayBudget",
                        new AccountMap(api.type())
                                .withEmailNotification(new EmailNotificationInfoMap(api.type()).on()
                                        .withPausedByDayBudget(invalidValue)),
                        new AxisError(25, AxisErrorDetails.FIELD_IN_STRUCTURE,
                                "PausedByDayBudget", "EmailNotification")),
                new TestCase("невозможный emailNotificationInfo.pausedByDayBudget",
                        new AccountMap(api.type())
                                .withSmsNotification(new SmsNotificationInfoMap(api.type()).on()
                                        .withPausedByDayBudgetSms(invalidValue)),
                        new AxisError(25, AxisErrorDetails.FIELD_IN_STRUCTURE,
                                "PausedByDayBudgetSms", "SmsNotification")),
        };
        return Arrays.stream(testCases).map(testCase -> new Object[] { testCase }).collect(Collectors.toList());
    }

    @BeforeClass
    public static void init() {
        api.userSteps.campaignSteps().verifyHasNonArchivedCampaign(LOGIN);
        accountId = api.userSteps.financeSteps().getAccountID(LOGIN);
        api.userSteps.financeSteps().verifySharedAccount(LOGIN);
    }

    @Test
    public void test() {
        AccountMap account = testCase.account.withAccountID(accountId);
        api.userSteps.financeSteps().shouldGetElementErrorOnUpdateAccount(account, testCase.expectedError);
    }
}
