package ru.yandex.autotests.directapi.finance.accountmanagement.update;

import java.util.Arrays;
import java.util.Collection;

import com.google.common.primitives.Ints;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.AccountMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * User: rockman
 * Date: 07.11.13
 */
@Aqua.Test
@Features(FinanceFeatures.ACCOUNT_MANAGEMENT_GET_UPDATE)
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class UpdateHasNoRightsTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameters
    public static Collection roleValues() {
        Object[][] data = new Object[][]{
                {FinanceLogins.AGENCY_TRANSFER_1, FinanceLogins.TRANSFER_CLIENT_1, FinanceLogins.GET_UPDATE_CLIENT_1},
                {Logins.MANAGER_DEFAULT, FinanceLogins.SERVICED_CLIENT1, FinanceLogins.GET_UPDATE_CLIENT_1},
                {FinanceLogins.GET_UPDATE_CLIENT_1, FinanceLogins.GET_UPDATE_CLIENT_1,
                        FinanceLogins.INVOICE_FREE_CLIENT},
                {FinanceLogins.INVOICE_FREE_CLIENT, FinanceLogins.INVOICE_FREE_CLIENT,
                        FinanceLogins.GET_UPDATE_CLIENT_1},
        };
        return Arrays.asList(data);
    }

    private String managerLogin;
    private String clientLogin;
    private String anotherLogin;
    private Account account;

    public UpdateHasNoRightsTest(String managerLogin, String clientLogin, String anotherLogin) {
        this.managerLogin = managerLogin;
        this.clientLogin = clientLogin;
        this.anotherLogin = anotherLogin;
        api.as(managerLogin);
    }

    @BeforeClass
    public static void initTest() {
        api.as(Logins.SUPER_LOGIN);
        api.userSteps.addModeratedCampaign(FinanceLogins.SERVICED_CLIENT1);
        api.userSteps.financeSteps().verifySharedAccount(FinanceLogins.SERVICED_CLIENT1);
        api.userSteps.clientFakeSteps().enableAgencyWallet(FinanceLogins.AGENCY_TRANSFER_1);
        api.as(FinanceLogins.AGENCY_TRANSFER_1).userSteps.createAccount(FinanceLogins.TRANSFER_CLIENT_1);
        api.userSteps.clientFakeSteps().enableToCreateSelfCampaigns(FinanceLogins.TRANSFER_CLIENT_1);
        api.as(Logins.AGENCY_TRANSFER_1);
        api.userSteps.campaignFakeSteps().resetWalletCreateDate(
                api.userSteps.addDraftCampaign(FinanceLogins.TRANSFER_CLIENT_1));
        api.as(Logins.AGENCY_TRANSFER_1).userSteps.financeSteps().verifySharedAccount(FinanceLogins.TRANSFER_CLIENT_1);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(FinanceLogins.GET_UPDATE_CLIENT_1);
        api.as(FinanceLogins.INVOICE_FREE_CLIENT).userSteps.addModeratedCampaign(FinanceLogins.INVOICE_FREE_CLIENT);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(FinanceLogins.INVOICE_FREE_CLIENT);
    }

    @Before
    public void getAccountInfo() {
        api.wsdl(APIPort_PortType.class);
        account = api.userSteps.financeSteps().getAccount(clientLogin);
        assumeThat("информация об общем счете получена", account, notNullValue());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6147")
    public void checkErrorWithNullLogin() {
        account.setAccountID(0);
        shouldGetErrorForAccountId(0);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6148")
    public void checkErrorWithWrongLogin() {
        account.setAccountID(42);
        shouldGetErrorForAccountId(42);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6149")
    public void checkErrorWithAnotherLogin() {
        api.as(anotherLogin);
        int accountId = api.userSteps.financeSteps().getAccountID(anotherLogin);
        api.as(managerLogin);
        account.setAccountID(accountId);
        shouldGetErrorForAccountId(accountId);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6146")
    public void getErrorRequestWithCampaignIDInAccountIDS() {
        long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(clientLogin);
        account.setAccountID(Ints.checkedCast(campaignID));
        shouldGetErrorForAccountId(campaignID);
    }

    private void shouldGetErrorForAccountId(long accountId) {
        api.userSteps.financeSteps().shouldGetErrorOnUpdateAccount(
                new AxisError(54, AxisErrorDetails.HAS_NO_RIGHTS_TO_EDIT_PARAMS_OF_ACCOUNT_ID2, accountId),
                new AccountMap<>(account));
    }
}
