package ru.yandex.autotests.directapi.finance.createinvoice;


import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * User: mariabye
 * Date: 24.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.CREATE_INVOICE)
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class CampaignIDValidateTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().wsdl(APIPort_PortType.class);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public String operatorLogin;

    @Parameterized.Parameter(value = 1)
    public String login;

    @Parameterized.Parameter(value = 2)
    public Currency currency;

    @Parameterized.Parameters(name = "Operator = {0}, Login = {1} ")
    public static Collection payments() {
        Object[][] data = new Object[][]{
                {FinanceLogins.AGENCY_INVOICE_1, FinanceLogins.INVOICE_SC_1, Currency.RUB},
                {FinanceLogins.INVOICE_FREE_CLIENT, FinanceLogins.INVOICE_FREE_CLIENT, Currency.RUB},
                {FinanceLogins.AGENCY_FREE_CLIENT_ACCOUNT, FinanceLogins.CLIENT_NO_ACCOUNT, Currency.RUB},
                {FinanceLogins.MANAGER_LOGIN, FinanceLogins.CLIENT_NO_ACCOUNT, Currency.RUB}
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(FinanceLogins.AGENCY_INVOICE_1);
        api.as(FinanceLogins.AGENCY_INVOICE_1).userSteps.createAccount(FinanceLogins.INVOICE_SC_1);
        api.as(FinanceLogins.INVOICE_FREE_CLIENT).userSteps.addModeratedCampaign(FinanceLogins.INVOICE_FREE_CLIENT);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(FinanceLogins.INVOICE_FREE_CLIENT);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6155")
    public void zeroCampaignID() {
        api.as(operatorLogin);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(1, AxisErrorDetails.EMPTY_STRING),
                new PayCampElementMap(api.type())
                        .withCurrency(currency)
                        .withCampaignID(0)
                        .withSum(20f));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6156")
    public void negativeCampaignID() {
        api.as(operatorLogin);
        long campaignID = api.userSteps.addDraftCampaign(login);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(1, AxisErrorDetails.EMPTY_STRING),
                new PayCampElementMap(api.type())
                        .withCurrency(currency)
                        .withCampaignID(-campaignID)
                        .withSum(20f));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6157")
    public void deletedCampaignID() {
        api.as(operatorLogin);
        long campaignID = api.userSteps.addDraftCampaign(login);
        api.userSteps.campaignSteps().campaignsDelete(login, campaignID);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(1, AxisErrorDetails.EMPTY_STRING),
                new PayCampElementMap(api.type())
                        .withCurrency(currency)
                        .withCampaignID(campaignID)
                        .withSum(20f));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6154")
    public void noRight() {
        api.as(operatorLogin);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(54, AxisErrorDetails.EMPTY_STRING),
                new PayCampElementMap(api.type())
                        .withCurrency(currency)
                        .withCampaignID(263)
                        .withSum(20f));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6158")
    public void firstCampaignNoRight() {
        api.as(operatorLogin);
        long campaignID = api.userSteps.addDraftCampaign(login);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(54, AxisErrorDetails.EMPTY_STRING),
                new PayCampElementMap(api.type())
                        .withCampaignID(263)
                        .withCurrency(currency)
                        .withSum(20f),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID)
                        .withCurrency(currency)
                        .withSum(20f));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6159")
    public void secondCampaignNoRight() {
        api.as(operatorLogin);
        long campaignID = api.userSteps.addDraftCampaign(login);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(54, AxisErrorDetails.EMPTY_STRING),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID)
                        .withCurrency(currency)
                        .withSum(20f),
                new PayCampElementMap(api.type())
                        .withCampaignID(263)
                        .withCurrency(currency)
                        .withSum(20f));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6160")
    public void createInvoicesForOneCampaign() {
        api.as(operatorLogin);
        long campaignID = api.userSteps.addModeratedCampaign(login);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(354, AxisErrorDetails.PAYCAMPELEMENT_CAMPAIGNID_MUST_BE_UNIQUE, campaignID),
                new PayCampElementMap(api.type())
                        .withCurrency(currency)
                        .withCampaignID(campaignID)
                        .withSum(20f),
                new PayCampElementMap(api.type())
                        .withCurrency(currency)
                        .withCampaignID(campaignID)
                        .withSum(20f));
    }
}


