package ru.yandex.autotests.directapi.finance.createinvoice;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;

/**
 * User: pavryabov
 * Date: 24.02.14
 */
@Aqua.Test
@Features(FinanceFeatures.CREATE_INVOICE)
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class ClientCreateInvoiceSumTest {
    protected LogSteps log = LogSteps.getLogger(this.getClass());

    private static final String AGENCY_LOGIN = Logins.AGENCY_RUB;
    private static final String CLIENT_LOGIN = Logins.SUBCLIENT_RUB;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private long campaignID1;
    private long campaignID2;
    private static Money minInvoiceAmount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT();

    @Parameterized.Parameter(value = 0)
    public String operator;

    @Parameterized.Parameters(name = "Operator = {0}")
    public static Collection logins() {
        Object[][] data = new Object[][]{
                {Logins.MANAGER_DEFAULT},
                {AGENCY_LOGIN}
        };
        return Arrays.asList(data);
    }

    @Before
    public void initTest() {
        campaignID1 = api.as(AGENCY_LOGIN).userSteps.addModeratedCampaign(CLIENT_LOGIN);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6161")
    public void createInvoiceForCampaignMinSumValue() {
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue()));
        api.userSteps.balanceSteps().operator(User.get(AGENCY_LOGIN)).requestAmountShouldBe(
                requestID, equalTo(minInvoiceAmount.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6162")
    public void createInvoiceForCampaignLowerThanMinSumValue() {
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(354, AxisErrorDetails.LOW_AMOUNT_SPECIFIED_FOR_CAMPAIGN_CUSTOM,
                        MoneyCurrency.get(Currency.RUB).getMinTransferRest()
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        Currency.RUB.value().toLowerCase(), campaignID1),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.getPrevious())
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6163")
    public void createInvoiceForCampaignOverMinSumValue() {
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.getNext()));
        api.userSteps.balanceSteps().operator(User.get(AGENCY_LOGIN)).requestAmountShouldBe(
                requestID, equalTo(minInvoiceAmount.getNext().floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6164")
    public void createInvoiceForCampaignsMinSumValue() {
        campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT_LOGIN);
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignID2);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount));
        api.userSteps.balanceSteps().operator(User.get(AGENCY_LOGIN)).requestAmountShouldBe(
                requestID, equalTo(minInvoiceAmount.multiply(2f).floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6165")
    public void createInvoiceForCampaignsOneLowerMinSumValue() {
        campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT_LOGIN);
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignID2);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(354, AxisErrorDetails.LOW_AMOUNT_SPECIFIED_FOR_CAMPAIGN_CUSTOM,
                        MoneyCurrency.get(Currency.RUB).getMinTransferRest()
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        Currency.RUB.value().toLowerCase(), campaignID2),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.getPrevious())
        );
    }
}
