package ru.yandex.autotests.directapi.finance.createinvoice;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;

/**
 * User: mariabye
 * Date: 11.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.CREATE_INVOICE)
@Tag(TagDictionary.TRUNK)
public class CreateInvoiceAgencyAccountTest {
    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final String AGENCY_LOGIN_1 = FinanceLogins.AGENCY_INVOICE_1;
    private static final String AGENCY_1_SUB_CLIENT_1_LOGIN = FinanceLogins.INVOICE_SC_1;
    private static final String AGENCY_1_SUB_CLIENT_2_LOGIN = FinanceLogins.INVOICE_SC_2;
    private static final String AGENCY_1_SUB_CLIENT_NO_ACCOUNT_LOGIN = FinanceLogins.INVOICE_SC_11;


    private static final Money minInvoiceAmount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT();

    @BeforeClass
    public static void initTest() {
        api.as(FinanceLogins.AGENCY_FREE_CLIENT_ACCOUNT).userSteps.createAccountForFreeClient(FinanceLogins.AGENCY_AND_FREE_CLIENT_YE);
        api.userSteps.clientFakeSteps().enableAgencyWallet(FinanceLogins.AGENCY_FREE_CLIENT_ACCOUNT);
        api.as(FinanceLogins.AGENCY_FREE_CLIENT_ACCOUNT)
                .userSteps.createAccount(FinanceLogins.AGENCY_AND_FREE_CLIENT_YE);
        api.userSteps.clientFakeSteps().enableAgencyWallet(AGENCY_LOGIN_1);
        api.as(AGENCY_LOGIN_1).userSteps.createAccount(AGENCY_1_SUB_CLIENT_1_LOGIN);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6171")
    public void agencyPaySubClientCampaign() {
        api.as(AGENCY_LOGIN_1);
        long campaignID = api.userSteps.addDraftCampaign(AGENCY_1_SUB_CLIENT_1_LOGIN);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB)
        );
        api.userSteps.balanceSteps().operator(User.get(AGENCY_LOGIN_1))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6177")
    public void agencyPaySubClientNoAccount() {
        api.as(AGENCY_LOGIN_1);
        long campaignID = api.userSteps.addDraftCampaign(AGENCY_1_SUB_CLIENT_NO_ACCOUNT_LOGIN);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue())
        );
        api.userSteps.balanceSteps().operator(User.get(AGENCY_LOGIN_1))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6169")
    public void agencyPayTwoCampaigns() {
        api.as(AGENCY_LOGIN_1);
        long campaignID1 = api.userSteps.addDraftCampaign(AGENCY_1_SUB_CLIENT_1_LOGIN);
        long campaignID2 = api.userSteps.addDraftCampaign(AGENCY_1_SUB_CLIENT_1_LOGIN);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB)
        );
        api.userSteps.balanceSteps().operator(User.get(AGENCY_LOGIN_1))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.multiply(2f).floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6170")
    public void agencyPayTwoSubClients() {
        api.as(AGENCY_LOGIN_1);
        long campaignID1 = api.userSteps.addDraftCampaign(AGENCY_1_SUB_CLIENT_1_LOGIN);
        long campaignID2 = api.userSteps.addDraftCampaign(AGENCY_1_SUB_CLIENT_2_LOGIN);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB)
        );
        api.userSteps.balanceSteps().operator(User.get(AGENCY_LOGIN_1))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.multiply(2f).floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6168")
    public void agencyPaySubClientWithAccountAndWithout() {
        api.as(AGENCY_LOGIN_1);
        long campaignID1 = api.userSteps.addDraftCampaign(AGENCY_1_SUB_CLIENT_NO_ACCOUNT_LOGIN);
        long campaignID2 = api.userSteps.addDraftCampaign(AGENCY_1_SUB_CLIENT_1_LOGIN);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB)
        );
        api.userSteps.balanceSteps().operator(User.get(AGENCY_LOGIN_1))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.multiply(2f).floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6178")
    public void accountIdInsteadOfCampaignID() {
        api.as(AGENCY_LOGIN_1);
        int accountID = api.userSteps.financeSteps().getAccountID(FinanceLogins.INVOICE_SC_1);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(new AxisError(1, null),
                new PayCampElementMap(api.type())
                        .withCampaignID(accountID)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue())
        );
    }
}
