package ru.yandex.autotests.directapi.finance.createinvoice;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;

/**
 * User: pavryabov
 * Date: 24.02.14
 */
@Aqua.Test
@Features(FinanceFeatures.CREATE_INVOICE)
@Tag(TagDictionary.TRUNK)
public class CreateInvoiceByClientTest {
    protected LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static final ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static String clientLogin = Logins.CLIENT_FOR_RUB;
    private static String clientWithDotInLogin = Logins.CLIENT_DOT_LOGIN_RUB_1;
    private static long campaignID1;
    private static long campaignID2;
    private static long anotherCampaignID = 263;
    private static long notModeratedCampaignID;
    private static Money minInvoiceAmount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT();

    @BeforeClass
    public static void initTest() {
        api.as(clientLogin);
        campaignID1 = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignID1);
        campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignID2);
        notModeratedCampaignID = api.userSteps.campaignSteps().addDefaultTextCampaign();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6187")
    public void couldCreateInvoiceOnModeratedCampaign() {
        api.as(clientLogin);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue()));
        api.userSteps.balanceSteps().operator(User.get(clientLogin)).requestAmountShouldBe(
                requestID, equalTo(minInvoiceAmount.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6188")
    public void couldNotCreateInvoiceForNoRightCampaign() {
        api.as(clientLogin);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(54, null),
                new PayCampElementMap(api.type())
                        .withCampaignID(anotherCampaignID)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue())
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6189")
    public void couldNotCreateInvoiceForNotModeratedCampaign() {
        api.as(clientLogin);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(354, AxisErrorDetails.CAMPAIGN_NOT_MODERATED, notModeratedCampaignID),
                new PayCampElementMap(api.type())
                        .withCampaignID(notModeratedCampaignID)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue())
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6194")
    public void testCreateInvoiceOneCampaignTwoInvoice() {
        api.as(clientLogin);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(354, AxisErrorDetails.PAYCAMPELEMENT_CAMPAIGNID_MUST_BE_UNIQUE),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue()),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue())
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6190")
    public void couldCreateInvoiceOnTwoCampaigns() {
        api.as(clientLogin);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue()),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue()));
        api.userSteps.balanceSteps().operator(User.get(clientLogin)).requestAmountShouldBe(
                requestID, equalTo(minInvoiceAmount.multiply(2f).floatValue()));
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6191")
    public void couldNotCreateInvoiceForTwoCampaignIfNoRightForOne() {
        api.as(clientLogin);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(54, null),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue()),
                new PayCampElementMap(api.type())
                        .withCampaignID(anotherCampaignID)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue())
        );
        log.info("Вызовем метод, поменяв местами элементы кампаний для оплаты");
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(54, null),
                new PayCampElementMap(api.type())
                        .withCampaignID(anotherCampaignID)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue()),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue())
        );
    }

    // https://jira.yandex-team.ru/browse/DIRECT-24803
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6192")
    public void couldNotCreateInvoiceForTwoCampaignOneNoRightCampaign() {
        api.as(Logins.SUPER_LOGIN, "holodilnikru");
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(54, null),
                new PayCampElementMap(api.type())
                        .withCampaignID(anotherCampaignID)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue()),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue())
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6193")
    public void createInvoiceWithDotLogins() {
        api.as(clientWithDotInLogin);
        long campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().makeCampaignModerated(campaignId);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignId)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceAmount.floatValue()));
        api.userSteps.balanceSteps().operator(User.get(clientWithDotInLogin)).requestAmountShouldBe(
                requestID, equalTo(minInvoiceAmount.floatValue()));
    }
}
