package ru.yandex.autotests.directapi.finance.createinvoice;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;

/**
 * User: mariabye
 * Date: 10.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.CREATE_INVOICE)
@Tag(TagDictionary.TRUNK)
public class CreateInvoiceClientAccountTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    static String login = FinanceLogins.INVOICE_FREE_CLIENT;
    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    Money minInvoiceAmount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT();

    @BeforeClass
    public static void initTest() {
        api.as(login).userSteps.addModeratedCampaign(login);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(login);
        api.as(login);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6205")
    public void createInvoiceForDraftCampaign() {
        long campaignID = api.userSteps.addDraftCampaign();
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB));

        api.userSteps.balanceSteps().operator(User.get(login))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6206")
    public void createInvoiceForModeratedCampaign() {
        long campaignID = api.userSteps.addModeratedCampaign();
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB));

        api.userSteps.balanceSteps().operator(User.get(login))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6207")
    public void createInvoiceForArchivedCampaign() {
        long campaignID = api.userSteps.addModeratedCampaign();
        api.userSteps.campaignSteps().campaignsSuspend(campaignID);
        api.userSteps.campaignSteps().campaignsArchive(campaignID);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB));

        api.userSteps.balanceSteps().operator(User.get(login))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6208")
    public void createInvoiceForCampaigns() {
        long campaignID1 = api.userSteps.addDraftCampaign();
        long campaignID2 = api.userSteps.addModeratedCampaign();
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withSum(minInvoiceAmount.divide(2f).floatValue())
                        .withCurrency(Currency.RUB)
        );
        api.userSteps.balanceSteps().operator(User.get(login)).requestAmountShouldBe(
                requestID, equalTo(minInvoiceAmount.add(minInvoiceAmount.divide(2f).floatValue()).floatValue()));
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6203")
    public void accountIdInsteadOfCampaignID() {
        int accountID = api.userSteps.financeSteps().getAccountID(login);
        api.userSteps.financeSteps()
                .shouldGetErrorOnCreateInvoice(new AxisError(1, null),
                        new PayCampElementMap(api.type())
                                .withCampaignID(accountID)
                                .withCurrency(Currency.RUB)
                                .withSum(minInvoiceAmount.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6204")
    public void servicedCampaignID() {
        long campaignID = api.as(FinanceLogins.MANAGER_LOGIN).userSteps
                .addModeratedCampaign(login);
        api.as(login);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB)
        );
        api.userSteps.balanceSteps().operator(User.get(login))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6209")
    public void createInvoiceForServicedAndSelfCampaign() {
        long campaignID1 = api.userSteps
                .addModeratedCampaign();
        long campaignID2 = api.as(FinanceLogins.MANAGER_LOGIN).userSteps
                .addModeratedCampaign(login);
        api.as(login);

        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withSum(minInvoiceAmount.floatValue())
                        .withCurrency(Currency.RUB)
        );
        api.userSteps.balanceSteps().operator(User.get(login))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceAmount.multiply(2.0).floatValue()));
    }
}
