package ru.yandex.autotests.directapi.finance.createinvoice;

import java.math.RoundingMode;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;

/**
 * Author pavryabov
 * Date 12.05.14
 */
@Aqua.Test
@Features(FinanceFeatures.CREATE_INVOICE)
@Tag(TagDictionary.TRUNK)
@Stories(ApiStories.CURRENCY)
public class CreateInvoiceCurrencyTest {
    protected LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static String noCurrencyClient = Logins.CLIENT_FREE_YE_DEFAULT;
    private static String rubClient = Logins.LOGIN_FOR_RUB;
    private static String usdClient = Logins.LOGIN_FOR_USD;
    private static String bynClient = Logins.BYN_CLIENT2;
    private static String bynServClient = Logins.SERV_CLIENT_BYN;
    private static long campaignIDRUB;
    private static long campaignIDUSD;
    private static long campaignIDNoCurrency;
    private static long campaignIDBYN;

    private static double bynRate;

    @BeforeClass
    public static void initTest() {
        api.as(Logins.AGENCY_YE_DEFAULT);
        campaignIDNoCurrency = api.userSteps.addActiveCampaign(noCurrencyClient, 1);
        api.as(rubClient);
        campaignIDRUB = api.userSteps.addActiveCampaign(rubClient, 1);
        api.as(usdClient);
        campaignIDUSD = api.userSteps.addActiveCampaign(usdClient, 1);
        api.as(bynClient);
        campaignIDBYN = api.userSteps.addActiveCampaign(bynClient, 1);
        bynRate = api.userSteps.balanceSteps().getCurrentCurrencyRate(Currency.BYN);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6216")
    public void createInvoiceRUBClientRUBInRequest() {
        api.as(rubClient).version(104);
        float sum = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT().floatValue() + 0.001f;
        log.info("Вызываем метод CreateInvoice для рублевого клиента с параметром Currency = RUB");
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignIDRUB)
                        .withSum(sum)
                        .withCurrency(Currency.RUB));
        api.userSteps.balanceSteps().operator(User.get(rubClient)).requestAmountShouldBe(
                requestID, equalTo(Money.valueOf(sum, Currency.RUB).setScale(2, RoundingMode.HALF_UP).floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6220")
    public void createInvoiceUSDClientUSDInRequest() {
        api.as(usdClient).version(104);
        float sum = MoneyCurrency.get(Currency.USD).getMinInvoiceAmount().addVAT().floatValue() + 0.001f;
        log.info("Вызываем метод CreateInvoice для долларового клиента с параметром Currency = USD");
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignIDUSD)
                        .withSum(sum)
                        .withCurrency(Currency.USD));
        api.userSteps.balanceSteps().operator(User.get(usdClient)).requestAmountShouldBe(
                requestID, equalTo(Money.valueOf(sum, Currency.USD).setScale(2, RoundingMode.HALF_UP).floatValue()));
    }

    @Test
    public void createInvoiceBYNClientBYNInRequest() {
        api.as(bynClient).version(104);
        float sum = MoneyCurrency.get(Currency.BYN).getMinInvoiceAmount().addVAT().floatValue() + 0.001f;
        log.info("Вызываем метод CreateInvoice для белорусского клиента с параметром Currency = BYN");
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignIDBYN)
                        .withSum(sum)
                        .withCurrency(Currency.BYN));
        log.info(bynRate);
        api.userSteps.balanceSteps().operator(User.get(bynClient)).requestAmountShouldBe(
                requestID, equalTo(Money.valueOf(sum, Currency.BYN).setScale(2, RoundingMode.HALF_UP)
                        .multiply(1.2f) // у BYN-клиентом нулевой график НДС в директе,
                        // но в балансе для этого продукта НДС 20% (https://st.yandex-team.ru/DIRECT-68481)
                        .floatValue()));
    }

    @Test
    public void expectErrorCreateInvoiceForBYNAndRUBClients() {
        api.as(Logins.MANAGER_DEFAULT).version(104);
        long mngCampaignIDBYN = api.userSteps.addActiveCampaign(bynServClient, 1);
        long mngCampaignIDRUB = api.userSteps.addActiveCampaign(rubClient, 1);
        float sum = MoneyCurrency.get(null).getMinInvoiceAmount().multiply(2f).addVAT(Currency.RUB).floatValue();
        log.info("Вызываем метод CreateInvoice для кампаний белорусского и рублевого клиентов");
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(245, AxisErrorDetails.NOT_ENABLE_INVOICE_CAMPAIGN_WITH_DIREFFERENT_CURRENCIES),
                new PayCampElementMap(api.type())
                        .withCampaignID(mngCampaignIDBYN)
                        .withSum(sum)
                        .withCurrency(Currency.RUB),
                new PayCampElementMap(api.type())
                        .withCampaignID(mngCampaignIDRUB)
                        .withSum(sum)
                        .withCurrency(Currency.RUB));
    }
}
