package ru.yandex.autotests.directapi.finance.createinvoice;

import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.APIPort_PortType;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;

/**
 * User: mariabye
 * Date: 24.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.CREATE_INVOICE)
@RunWith(Parameterized.class)
public class SumForAccountValidateTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    Money minInvoiceSumRUB = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT();

    static String priceMessage = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount()
            .stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED);

    @Parameterized.Parameter(value = 0)
    public String operatorLogin;

    @Parameterized.Parameter(value = 1)
    public String login;

    @Parameterized.Parameter(value = 2)
    public String balanceOperator;

    @Parameterized.Parameters(name = "Operator = {0}, Login = {1} ")
    public static Collection payments() {
        Object[][] data = new Object[][]{
                {FinanceLogins.AGENCY_INVOICE_1, FinanceLogins.INVOICE_SC_1, FinanceLogins.AGENCY_INVOICE_1},
                {FinanceLogins.INVOICE_FREE_CLIENT, FinanceLogins.INVOICE_FREE_CLIENT,
                        FinanceLogins.INVOICE_FREE_CLIENT},
                {FinanceLogins.MANAGER_LOGIN, FinanceLogins.INVOICE_FREE_CLIENT, FinanceLogins.INVOICE_FREE_CLIENT}
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void initTest() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(FinanceLogins.AGENCY_INVOICE_1);
        api.as(FinanceLogins.AGENCY_INVOICE_1).userSteps.createAccount(FinanceLogins.INVOICE_SC_1);
        api.as(FinanceLogins.INVOICE_FREE_CLIENT).userSteps.addModeratedCampaign(FinanceLogins.INVOICE_FREE_CLIENT);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(FinanceLogins.INVOICE_FREE_CLIENT);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6263")
    public void zeroSumCurrency() {
        api.as(operatorLogin);
        int accountID = api.userSteps.financeSteps().getAccountID(login);
        long campaignID = api.userSteps.addModeratedCampaign(login);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(354, AxisErrorDetails.INVALID_AMOUNT_SPECIFIED_FOR_ACCOUNT, accountID),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID)
                        .withCurrency(Currency.RUB)
                        .withSum(0f)
        );
    }

    @Test

    @ru.yandex.qatools.allure.annotations.TestCaseId("6265")
    public void lowSumWithCurrency() {
        api.as(operatorLogin);
        int accountID = api.userSteps.financeSteps().getAccountID(login);
        long campaignID = api.userSteps.addModeratedCampaign(login);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(354, AxisErrorDetails.LOW_CURRENCY_PAYMENT_AMOUNT_FOR_ACCOUNT,
                        priceMessage, MoneyCurrency.get(Currency.RUB).getApiDescription(), accountID),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceSumRUB.getPrevious())
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6266")
    public void minTotalSum() {
        api.as(operatorLogin);

        long campaignID1 = api.userSteps.addModeratedCampaign(login);
        long campaignID2 = api.userSteps.addModeratedCampaign(login);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceSumRUB.divide(2)),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceSumRUB.divide(2))
        );
        api.userSteps.balanceSteps().operator(User.get(balanceOperator))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceSumRUB.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6267")
    public void zeroAndNormalSum() {
        api.as(operatorLogin);
        long campaignID1 = api.userSteps.addModeratedCampaign(login);
        long campaignID2 = api.userSteps.addModeratedCampaign(login);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(0f),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceSumRUB)
        );
        api.userSteps.balanceSteps().operator(User.get(balanceOperator))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceSumRUB.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6268")
    public void negativeAndNormalSum() {
        api.as(operatorLogin);
        long campaignID1 = api.userSteps.addModeratedCampaign(login);
        long campaignID2 = api.userSteps.addModeratedCampaign(login);
        String requestID = api.userSteps.financeSteps().getRequestID(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(-1.0f),
                new PayCampElementMap(api.type())
                        .withCurrency(Currency.RUB)
                        .withCampaignID(campaignID2)
                        .withSum(minInvoiceSumRUB.add(1.0f))
        );
        api.userSteps.balanceSteps().operator(User.get(balanceOperator))
                .requestAmountShouldBe(requestID, equalTo(minInvoiceSumRUB.floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6269")
    public void lowTotalSum() {
        api.as(operatorLogin);
        int accountID = api.userSteps.financeSteps().getAccountID(login);
        long campaignID1 = api.userSteps.addModeratedCampaign(login);
        long campaignID2 = api.userSteps.addModeratedCampaign(login);
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoice(
                new AxisError(354, AxisErrorDetails.LOW_CURRENCY_PAYMENT_AMOUNT_FOR_ACCOUNT,
                        priceMessage, MoneyCurrency.get(Currency.RUB).getApiDescription(), accountID),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceSumRUB.divide(2).add(-2.0f)),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceSumRUB.divide(2).add(1.0f))
        );
    }
}


