package ru.yandex.autotests.directapi.finance.enablesharedaccount;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.CampaignsRecord;
import ru.yandex.autotests.direct.db.models.jooq.ppc.tables.records.StrategiesRecord;
import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.common.api45.CreateNewSubclientResponse;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.EnableSharedAccountRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * User: mariabye
 * Date: 05.11.13
 */
@Aqua.Test
@Features(FinanceFeatures.ENABLE_SHARED_ACCOUNT)
public class EnableSharedAccountLoginTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    public static String agencyLogin = FinanceLogins.AGENCY_ENABLE_SHARED_ACCOUNT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(FinanceLogins.AGENCY_ENABLE_SHARED_ACCOUNT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void sharedAccountFlagOn() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(agencyLogin);
        api.userSteps.clientFakeSteps().enableToCreateSubClients(agencyLogin);
    }

    private void disableSharedAccount(Long campaignId, int shard) {
        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard);
        CampaignsRecord campaiangs = jooqDbSteps.campaignsSteps().getCampaignById(campaignId);
        campaiangs.setWalletCid(0L);
        jooqDbSteps.campaignsSteps().updateCampaigns(campaiangs);

        api.userSteps.campaignFakeSteps().resetWalletCreateDate(campaignId);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6283")
    public void emptyLogin() {
        api.userSteps.financeSteps().shouldGetErrorOn(
                Method.ENABLE_SHARED_ACCOUNT,
                new EnableSharedAccountRequestMap(api.type()).withLogins(""),
                new AxisError(251, AxisErrorDetails.ANY_FIELD, "Login"));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6290")
    public void agencyLogin() {
        api.userSteps.financeSteps().shouldGetErrorsOnEnableSharedAccount(
                new EnableSharedAccountRequestMap(api.type()).withLogins("at-direct-ag-full"),
                new AxisError(54, AxisErrorDetails.NOT_AGENCY_SUB_CLIENT));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6282")
    public void managerLogin() {
        api.userSteps.financeSteps().shouldGetErrorsOnEnableSharedAccount(
                new EnableSharedAccountRequestMap(api.type()).withLogins("at-direct-mngr-full"),
                new AxisError(54, AxisErrorDetails.NOT_AGENCY_SUB_CLIENT));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6281")
    public void noRightLogin() {
        api.userSteps.financeSteps().shouldGetErrorsOnEnableSharedAccount(
                new EnableSharedAccountRequestMap(api.type()).withLogins("at-direct-ag-client"),
                new AxisError(54, AxisErrorDetails.NOT_AGENCY_SUB_CLIENT));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6284")
    public void selfAgencyLogin() {
        api.userSteps.financeSteps().shouldGetErrorsOnEnableSharedAccount(
                new EnableSharedAccountRequestMap(api.type()).withLogins(agencyLogin),
                new AxisError(54, AxisErrorDetails.NOT_AGENCY_SUB_CLIENT));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6285")
    public void walletNotEnabledCampaignExists() {
        CreateNewSubclientResponse subClient = api.userSteps.clientSteps()
                .createNewAgencySubClient("sh", agencyLogin, Currency.RUB);

        int shard = api.userSteps.clientFakeSteps().getUserShard(subClient.getLogin());
        disableSharedAccount(
                api.userSteps.addDraftCampaign(subClient.getLogin()), shard);

        int accountID = api.userSteps.financeSteps().enableAndGetSharedAccount(subClient.getLogin());
        Account account = api.userSteps.financeSteps().getAccount(accountID);
        assertThat(account.getLogin(), equalTo(subClient.getLogin().toLowerCase()));
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6286")
    public void newSubclientFlagOn() {
        CreateNewSubclientResponse subClient = api.userSteps.clientSteps()
                .createNewAgencySubClient("sh", agencyLogin, Currency.RUB);

        int shard = api.userSteps.clientFakeSteps().getUserShard(subClient.getLogin());
        disableSharedAccount(api.userSteps.addModeratedCampaign(subClient.getLogin()), shard);
        int accountID = api.userSteps.financeSteps().enableAndGetSharedAccount(subClient.getLogin());
        Account account = api.userSteps.financeSteps().getAccount(accountID);
        assertThat(account.getLogin(), equalTo(subClient.getLogin().toLowerCase()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6287")
    public void newSubclientWithDotFlagOn() {
        CreateNewSubclientResponse subClient =
                api.userSteps.clientSteps().createNewAgencySubClient("sh.s", agencyLogin, Currency.RUB);

        int shard = api.userSteps.clientFakeSteps().getUserShard(subClient.getLogin());
        disableSharedAccount(api.userSteps.addModeratedCampaign(subClient.getLogin()), shard);
        int accountID = api.userSteps.financeSteps().enableAndGetSharedAccount(subClient.getLogin());
        Account account = api.userSteps.financeSteps().getAccount(accountID);
        assertThat(account.getLogin(), equalTo(subClient.getLogin().toLowerCase()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6288")
    public void newSubclientFlagOnTwice() {
        CreateNewSubclientResponse subClient = api.userSteps.clientSteps()
                .createNewAgencySubClient("sh", agencyLogin, Currency.RUB);

        int shard = api.userSteps.clientFakeSteps().getUserShard(subClient.getLogin());
        disableSharedAccount(api.userSteps.addModeratedCampaign(subClient.getLogin()), shard);
        int accountID = api.userSteps.financeSteps().enableAndGetSharedAccount(subClient.getLogin());
        Account account = api.userSteps.financeSteps().getAccount(accountID);
        assumeThat("в информации о кошельке правильный логин",
                account.getLogin(), equalTo(subClient.getLogin().toLowerCase()));

        api.userSteps.financeSteps().shouldGetErrorsOnEnableSharedAccount(
                new EnableSharedAccountRequestMap(api.type()).withLogins(subClient.getLogin()),
                new AxisError(519, AxisErrorDetails.UNABLE_TO_PERFORM_TRANSACTION_ACCOUNT_HAS_BEEN_ON));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6289")
    public void noDirectClient() {
        api.userSteps.financeSteps().shouldGetErrorsOnEnableSharedAccount(
                new EnableSharedAccountRequestMap(api.type()).withLogins(Logins.NO_DIRECT_LOGIN),
                new AxisError(54, AxisErrorDetails.NOT_AGENCY_SUB_CLIENT));
    }

    @Test
    public void minorRepresentorHavePermissionsToActivate() {
        // TESTIRT-10025 не главный представитель клиента может подключать ОС
        String agencyLogin = FinanceLogins.AGENCY_WITH_RESTRICTED_REPRESENTOR;
        String restrictedRepresentor = FinanceLogins.AGENCY_RESTRICTED_REPRESENTOR;
        api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().enableAgencyWallet(agencyLogin);
        api.as(Logins.SUPER_LOGIN).userSteps.clientFakeSteps().enableToCreateSubClients(agencyLogin);

        CreateNewSubclientResponse response = api.as(restrictedRepresentor).userSteps.clientSteps()
                .createNewAgencySubClient("at-sc-right-", restrictedRepresentor, Currency.RUB);
        String subClient = response.getLogin();
        assumeThat("клиент создался", subClient.length(), greaterThan(0));

        int shard = api.userSteps.clientFakeSteps().getUserShard(subClient);
        disableSharedAccount(
                api.userSteps.campaignSteps().addDefaultTextCampaign(subClient), shard);

        api.as(restrictedRepresentor).userSteps.financeSteps().enableSharedAccount(subClient);
        api.as(restrictedRepresentor).userSteps.financeSteps().shouldGetErrorsOnEnableSharedAccount(
                new EnableSharedAccountRequestMap(api.type()).withLogins(subClient),
                new AxisError(519, AxisErrorDetails.UNABLE_TO_PERFORM_TRANSACTION_ACCOUNT_HAS_BEEN_ON));
    }

    @Test
    public void strategyIdTest() {
        CreateNewSubclientResponse subClient =
                api.userSteps.clientSteps().createNewAgencySubClient("sh.s", agencyLogin, Currency.RUB);

        int shard = api.userSteps.clientFakeSteps().getUserShard(subClient.getLogin());
        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard);
        disableSharedAccount(api.userSteps.addModeratedCampaign(subClient.getLogin()), shard);
        int accountID = api.userSteps.financeSteps().enableAndGetSharedAccount(subClient.getLogin());
        CampaignsRecord campaignsRecord = jooqDbSteps.campaignsSteps().getCampaignById((long) accountID);
        Long strategyId = campaignsRecord.getStrategyId();
        assertThat("кошелек создался без стратегии",
                strategyId, equalTo(0L));
    }

    @Test
    public void walletCidInStrategyAfterEnbaleWalletTest() {
        CreateNewSubclientResponse subClient =
                api.userSteps.clientSteps().createNewAgencySubClient("sh.s", agencyLogin, Currency.RUB);

        int shard = api.userSteps.clientFakeSteps().getUserShard(subClient.getLogin());
        DirectJooqDbSteps jooqDbSteps = api.userSteps.getDirectJooqDbSteps().useShard(shard);
        Long cid = api.userSteps.addModeratedCampaign(subClient.getLogin());
        disableSharedAccount(cid, shard);
        CampaignsRecord campaignsRecord = jooqDbSteps.campaignsSteps().getCampaignById(cid);
        Long strategyId = campaignsRecord.getStrategyId();
        StrategiesRecord strategiesRecord = jooqDbSteps.strategiesSteps().getStrategiesRecord(strategyId);
        strategiesRecord.setWalletCid(0L);
        jooqDbSteps.strategiesSteps().updateStrategies(strategiesRecord);
        int accountID = api.userSteps.financeSteps().enableAndGetSharedAccount(subClient.getLogin());
        strategiesRecord = jooqDbSteps.strategiesSteps().getStrategiesRecord(strategyId);
        assertThat("wallet_cid обновился в стратегии", strategiesRecord.getWalletCid(), equalTo((long) accountID));
    }
}
