package ru.yandex.autotests.directapi.finance.enablesharedaccount;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45mng.Account;
import ru.yandex.autotests.directapi.common.api45mng.CreateNewSubclientResponse;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.sharedaccount.EnableSharedAccountRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.core.IsNull.notNullValue;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * User: mariabye
 * Date: 05.11.13
 * <p>
 * https://jira.yandex-team.ru/browse/DIRECT-26208
 */
@Aqua.Test
@Features(FinanceFeatures.ENABLE_SHARED_ACCOUNT)
public class EnableSharedAccountOnCampaignCreationTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    public static String agencyLogin = FinanceLogins.AGENCY_ENABLE_SHARED_ACCOUNT;
    public static String managerLogin = Logins.MANAGER_GEO;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class).as(agencyLogin);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void sharedAccountFlagOn() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(agencyLogin);
        api.userSteps.clientFakeSteps().enableToCreateSubClients(agencyLogin);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6292")
    public void newRubSubclientFlagOn() {
        api.as(agencyLogin);
        CreateNewSubclientResponse subClient = api.userSteps.clientSteps()
                .createNewAgencySubClient("create", agencyLogin, Currency.RUB);
        api.userSteps.addDraftCampaign(subClient.getLogin());
        Account account = api.userSteps.financeSteps().getAccount(subClient.getLogin());
        assertThat(account.getLogin(), equalTo(subClient.getLogin().toLowerCase()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6293")
    public void createSecondCampaign() {
        api.as(agencyLogin);
        CreateNewSubclientResponse subClient = api.userSteps.clientSteps()
                .createNewAgencySubClient("sh-create", agencyLogin, Currency.RUB);
        api.userSteps.addDraftCampaign(subClient.getLogin());
        api.userSteps.addDraftCampaign(subClient.getLogin());
        Account account = api.userSteps.financeSteps().getAccount(subClient.getLogin());
        assertThat(account.getLogin(), equalTo(subClient.getLogin().toLowerCase()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6291")
    public void newSubclientWithDotFlagOn() {
        api.as(agencyLogin);
        CreateNewSubclientResponse subClient = api.userSteps.clientSteps()
                .createNewAgencySubClient("sh.create", agencyLogin, Currency.RUB);
        api.userSteps.addDraftCampaign(subClient.getLogin());
        Account account = api.userSteps.financeSteps().getAccount(subClient.getLogin());
        assertThat(account.getLogin(), equalTo(subClient.getLogin().toLowerCase()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6294")
    public void callEnableSharedAccountAfterCampaignCreation() {
        api.as(agencyLogin);
        CreateNewSubclientResponse subClient = api.userSteps.clientSteps()
                .createNewAgencySubClient("sh-create", agencyLogin, Currency.RUB);
        api.userSteps.addDraftCampaign(subClient.getLogin());

        api.userSteps.financeSteps().shouldGetErrorsOnEnableSharedAccount(
                new EnableSharedAccountRequestMap(api.type()).withLogins(subClient.getLogin()),
                new AxisError(519, AxisErrorDetails.UNABLE_TO_PERFORM_TRANSACTION_ACCOUNT_HAS_BEEN_ON));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6295")
    public void noAccountIfNoCampaign() {
        api.as(agencyLogin);
        CreateNewSubclientResponse subClient = api.userSteps.clientSteps()
                .createNewAgencySubClient("sh-create", agencyLogin, Currency.RUB);


        api.userSteps.financeSteps().shouldGetErrorsOnAccountManagementGet(
                subClient.getLogin(),
                new AxisError(515, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6296")
    public void newServicedClient() {
        api.as(managerLogin);
        CreateNewSubclientResponse subClient =
                api.userSteps.clientSteps().createServicedClient("subClient", managerLogin);
        api.userSteps.addDraftCampaign(subClient.getLogin());
        Account account = api.userSteps.financeSteps().getAccount(subClient.getLogin());
        assertThat(account.getLogin(), equalTo(subClient.getLogin().toLowerCase()));
    }

    @Test
    public void createCampaignAfterDeleteAllVersion5() {
        api.as(agencyLogin);
        CreateNewSubclientResponse subClient = api.userSteps.clientSteps()
                .createNewAgencySubClient("create", agencyLogin, Currency.RUB);
        Long camapginId = api.userSteps.campaignSteps().addDefaultTextCampaign(subClient.getLogin());
        Account account = api.userSteps.financeSteps().getAccount(subClient.getLogin());
        assumeThat("у клиента включился ОС", account.getLogin(), equalTo(subClient.getLogin().toLowerCase()));
        int accountId = account.getAccountID();
        api.userSteps.campaignSteps().campaignsDelete(subClient.getLogin(), camapginId);
        api.userSteps.financeSteps().shouldGetErrorsOnAccountManagementGet(
                subClient.getLogin(),
                new AxisError(515, AxisErrorDetails.EMPTY_STRING));
        api.userSteps.campaignSteps().addDefaultTextCampaign(subClient.getLogin());
        account = api.userSteps.financeSteps().getAccount(subClient.getLogin());
        assumeThat("вернулась информация по ОС клиента", account, notNullValue());
        assertThat("у клиента включился ОС", account.getAccountID(), equalTo(accountId));
    }
}
