package ru.yandex.autotests.directapi.finance.paycampaigns;

import java.math.BigDecimal;

import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.PayCampaignsInfoMap;
import ru.yandex.autotests.directapi.model.finances.PayMethod;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * User: mariabye
 * Date: 17.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.PAY_CAMPAIGNS)
@Tag(TagDictionary.TRUNK)
public class PayCampaignsBankAccountTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static final String AGENCY_LOGIN = FinanceLogins.AGENCY_DEPOSIT_1;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(AGENCY_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final Money MIN_INVOICE_AMOUNT = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount();
    private static final String AGENCY_SUBCLIENT_1_LOGIN = FinanceLogins.DEPOSIT_SC_6;
    private static final String AGENCY_SUBCLIENT_2_LOGIN = FinanceLogins.DEPOSIT_SC_7;

    private static int accountID1;
    private static int accountID2;
    private static String agencyContract;

    @BeforeClass
    public static void getAccountIDs() {
        api.userSteps.clientFakeSteps().enableAgencyWallet(AGENCY_LOGIN);
        api.as(AGENCY_LOGIN).userSteps.createAccount(FinanceLogins.DEPOSIT_SC_2);
        api.userSteps.createAccount(FinanceLogins.DEPOSIT_SC_6);
        api.userSteps.createAccount(FinanceLogins.DEPOSIT_SC_7);
        accountID1 = api.userSteps.financeSteps().getAccountID(AGENCY_SUBCLIENT_1_LOGIN);
        accountID2 = api.userSteps.financeSteps().getAccountID(AGENCY_SUBCLIENT_2_LOGIN);
        api.userSteps.financeSteps().verifyCampaignExistsInBalance(accountID1);
        api.userSteps.financeSteps().verifyCampaignExistsInBalance(accountID2);
        api.userSteps.balanceSteps().synchronizeWithBalance(accountID1);
        api.userSteps.balanceSteps().synchronizeWithBalance(accountID2);
        agencyContract = User.get(AGENCY_LOGIN).getExternalContractID();

        api.userSteps.clientFakeSteps().enableAgencyWallet(FinanceLogins.AGENCY_DEPOSIT_2);
        api.as(FinanceLogins.AGENCY_DEPOSIT_2).userSteps.createAccount(FinanceLogins.DEPOSIT_SC_21);
        api.userSteps.financeSteps().verifyAccountExistsInBalance(FinanceLogins.DEPOSIT_SC_21);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6371")
    public void paySubClientCampaign() {
        log.info("case1 start - " + System.currentTimeMillis());
        api.as(AGENCY_LOGIN);
        long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(AGENCY_SUBCLIENT_1_LOGIN);
        Account account = api.userSteps.financeSteps().getAccount(AGENCY_SUBCLIENT_1_LOGIN);

        api.userSteps.financeSteps().payCampaignsBank(
                agencyContract,
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID)
                        .withCurrency(Currency.RUB)
                        .withSum(MIN_INVOICE_AMOUNT.addVAT().floatValue())
        );


        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps().accountAmountChangedOnSum(account, MIN_INVOICE_AMOUNT.floatValue()),
                equalTo(true));
        log.info("case1 finish - " + System.currentTimeMillis());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6373")
    public void paySubClientCampaigns() {
        log.info("case2 start - " + System.currentTimeMillis());
        api.as(AGENCY_LOGIN);
        long campaignID1 = api.userSteps.campaignSteps().addDefaultTextCampaign(AGENCY_SUBCLIENT_1_LOGIN);
        long campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign(AGENCY_SUBCLIENT_1_LOGIN);
        Account account = api.userSteps.financeSteps().getAccount(AGENCY_SUBCLIENT_1_LOGIN);

        api.userSteps.financeSteps().payCampaignsBank(
                agencyContract,
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(MIN_INVOICE_AMOUNT.addVAT().floatValue()),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withCurrency(Currency.RUB)
                        .withSum(MIN_INVOICE_AMOUNT.add(10f).addVAT().floatValue())
        );


        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps().accountAmountChangedOnSum(
                        account, MIN_INVOICE_AMOUNT.multiply(2f).add(10f).floatValue()), equalTo(true));
        log.info("case2 finish - " + System.currentTimeMillis());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6374")
    public void paySubClientsCampaigns() {
        log.info("case3 start - " + System.currentTimeMillis());
        api.as(AGENCY_LOGIN);
        long campaignID1 = api.userSteps.campaignSteps().addDefaultTextCampaign(AGENCY_SUBCLIENT_1_LOGIN);
        long campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign(AGENCY_SUBCLIENT_2_LOGIN);

        Account account1 = api.userSteps.financeSteps().getAccount(AGENCY_SUBCLIENT_1_LOGIN);
        Money account1AmountBefore = api.userSteps.financeSteps().getAccountAmount(account1.getAccountID());
        assumeThat("Общий счёт найден и получена сумма на нём", account1AmountBefore, is(notNullValue()));

        Account account2 = api.userSteps.financeSteps().getAccount(AGENCY_SUBCLIENT_2_LOGIN);
        Money account2AmountBefore = api.userSteps.financeSteps().getAccountAmount(account2.getAccountID());
        assumeThat("Общий счёт найден и получена сумма на нём", account2AmountBefore, is(notNullValue()));

        api.userSteps.financeSteps().payCampaignsBank(
                agencyContract,
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(MIN_INVOICE_AMOUNT.addVAT().floatValue()),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withCurrency(Currency.RUB)
                        .withSum(MIN_INVOICE_AMOUNT.add(10f).addVAT().floatValue())
        );

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps().accountAmountChangedOnSum(
                        account1.getAccountID(), account1AmountBefore, MIN_INVOICE_AMOUNT));
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps().accountAmountChangedOnSum(
                        account2.getAccountID(), account2AmountBefore, MIN_INVOICE_AMOUNT.add(10f)));
        log.info("case3 finish - " + System.currentTimeMillis());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6370")
    public void paySubClientsCampaignsWithAccountAndWithout() {
        api.as(FinanceLogins.AGENCY_DEPOSIT_2);

        long campaignID1 = api.userSteps.campaignSteps().addDefaultTextCampaign(FinanceLogins.DEPOSIT_SC_21);
        Account account = api.userSteps.financeSteps().getAccount(FinanceLogins.DEPOSIT_SC_21);
        api.userSteps.balanceSteps().synchronizeWithBalance(account.getAccountID());
        Money accountAmountBefore = api.userSteps.financeSteps().getAccountAmount(account.getAccountID());
        assumeThat("Общий счёт найден и получена сумма на нём", accountAmountBefore, is(notNullValue()));

        long campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign(FinanceLogins.DEPOSIT_SC_3);
        CampaignGetItem campaign = api.userSteps.campaignSteps().getCampaign(FinanceLogins.DEPOSIT_SC_3, campaignID2);
        Money campaignAmountBefore = Money.valueOf(campaign.getFunds().getCampaignFunds().getSum(), Currency.RUB)
                .subtractVAT()
                .divide(BigDecimal.valueOf(1000000L));

        api.userSteps.financeSteps().payCampaignsBank(
                User.get(FinanceLogins.AGENCY_DEPOSIT_2).getExternalContractID(),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(MIN_INVOICE_AMOUNT.addVAT().floatValue()),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withCurrency(Currency.RUB)
                        .withSum(MIN_INVOICE_AMOUNT.add(10f).addVAT().floatValue())
        );

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.financeSteps().accountAmountChanged(account));
        Money accountAmountAfter = api.userSteps.financeSteps().getAccountAmount(account.getAccountID());
        assertThat("Неверная сумма на счёте",
                accountAmountAfter, equalTo(accountAmountBefore.add(MIN_INVOICE_AMOUNT)));

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(FinanceLogins.DEPOSIT_SC_3, campaign));
        campaign = api.userSteps.campaignSteps().getCampaign(FinanceLogins.DEPOSIT_SC_3, campaignID2);
        Money campaignAmountAfter = Money.valueOf(campaign.getFunds().getCampaignFunds().getSum(), Currency.RUB)
                .subtractVAT()
                .divide(BigDecimal.valueOf(1000000L));
        assertThat("Неверная сумма на кампании",
                campaignAmountAfter, equalTo(campaignAmountBefore.add(MIN_INVOICE_AMOUNT.add(10f))));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6372")
    public void payAccountID() {
        api.as(AGENCY_LOGIN);

        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(1),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID(agencyContract)
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(accountID1)
                                .withCurrency(Currency.RUB)
                                .withSum(MIN_INVOICE_AMOUNT.addVAT().floatValue()))
        );
    }
}
