package ru.yandex.autotests.directapi.finance.paycampaigns;

import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.PayCampaignsInfoMap;
import ru.yandex.autotests.directapi.model.finances.PayMethod;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.directapi.test.ClientIdLock;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang.StringUtils.capitalize;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;

/**
 * Author pavryabov
 * Date 15.04.14
 */
@Aqua.Test
@Features(FinanceFeatures.PAY_CAMPAIGNS)
public class PayCampaignsBankTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static final String AGENCY_LOGIN = Logins.AGENCY_RUB;
    private static final String CLIENT_LOGIN = Logins.SUBCLIENT_RUB;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(AGENCY_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static ClientIdLock lock;

    private static Money sum = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT();
    private static Money SUM_NO_VAT = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount();
    private static long activeCampaignID1;
    private static long activeCampaignID2;
    private static long deletedCampaignID;
    private String defaultContract =
            api.userSteps.balanceSteps().getDefaultContract(User.get(AGENCY_LOGIN).getClientID(), 1);

    @BeforeClass
    public static void createCampaigns() {
        lock = new ClientIdLock("CreditLimits" + AGENCY_LOGIN);
        lock.acquire(3);
        api.as(AGENCY_LOGIN);
        activeCampaignID1 = api.userSteps.addActiveCampaign(CLIENT_LOGIN, 1);
        activeCampaignID2 = api.userSteps.addActiveCampaign(CLIENT_LOGIN, 1);
        deletedCampaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(CLIENT_LOGIN);
        api.userSteps.campaignSteps().campaignsDelete(CLIENT_LOGIN, deletedCampaignID);
    }

    @AfterClass
    public static void releaseLock() {
        lock.release();
    }

    /**
     * оплата чужой кампании
     */

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6375")
    public void testPayNotMyCampaign() {
        api.as(AGENCY_LOGIN);
        log.info("Попытка оплатить кампанию, на которую нет прав (263) вместе с кампанией, на которую есть права");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(54, null),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID(defaultContract)
                        .withPayments(new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(263)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum))
        );
    }

    @Test
    // добавлены проверки https://jira.yandex-team.ru/browse/DIRECT-12316
    @ru.yandex.qatools.allure.annotations.TestCaseId("6376")
    public void testPayCampaign() {
        api.as(AGENCY_LOGIN);
        log.info("Оплата кампании по кредитному договору на сумму: " + sum.floatValue());
        CampaignGetItem campaign = api.userSteps.campaignSteps().getCampaign(CLIENT_LOGIN, activeCampaignID1);
        int response = api.protocol(ProtocolType.SOAP).userSteps.financeSteps().payCampaignsBank(
                defaultContract,
                new PayCampElementMap(api.type())
                        .withCampaignID(activeCampaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(sum));
        assertThat("Метод PayCampaigns вернул неуспешный код завершения операции", response, equalTo(1));
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChangedOnFunds(CLIENT_LOGIN, campaign, SUM_NO_VAT));
    }

    @Test
    // добавлены проверки https://jira.yandex-team.ru/browse/DIRECT-12316
    @ru.yandex.qatools.allure.annotations.TestCaseId("6377")
    public void testPayCampaignJSON() {
        api.as(AGENCY_LOGIN);
        log.info("Оплата кампании по кредитному договору на сумму: " + sum);
        CampaignGetItem campaign = api.userSteps.campaignSteps().getCampaign(CLIENT_LOGIN, activeCampaignID1);
        int response = api.protocol(ProtocolType.JSON).userSteps.financeSteps().payCampaignsBank(
                defaultContract,
                new PayCampElementMap(api.type())
                        .withCampaignID(activeCampaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(sum));
        assertThat("Метод PayCampaigns вернул неуспешный код завершения операции", response, equalTo(1));
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChangedOnFunds(CLIENT_LOGIN, campaign, SUM_NO_VAT));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6378")
    public void testPayCampaignZeroSumToPay() {
        api.as(AGENCY_LOGIN);
        Money spentBefore = api.userSteps.financeSteps().getContractSpent(defaultContract);
        log.info("Оплата кампании по кредитному договору на сумму: 0 уе");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(354, AxisErrorDetails.INVALID_AMOUNT_SPECIFIED_FOR_CAMPAIGN, activeCampaignID1),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID(defaultContract)
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(0f)));
        Money spentAfter = api.userSteps.financeSteps().getContractSpent(defaultContract);
        assertThat("Неверные параметры кредитных остатков", spentAfter, equalTo(spentBefore));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6379")
    public void testPayCampaigns() {
        api.as(AGENCY_LOGIN);
        log.info("Оплата двух кампаний на одинаковую сумму " + sum);
        CampaignGetItem campaign1 = api.userSteps.campaignSteps().getCampaign(CLIENT_LOGIN, activeCampaignID1);
        CampaignGetItem campaign2 = api.userSteps.campaignSteps().getCampaign(CLIENT_LOGIN, activeCampaignID2);
        int response = api.userSteps.financeSteps().payCampaignsBank(
                defaultContract,
                new PayCampElementMap(api.type())
                        .withCampaignID(activeCampaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(sum),
                new PayCampElementMap(api.type())
                        .withCampaignID(activeCampaignID2)
                        .withCurrency(Currency.RUB)
                        .withSum(sum));
        assertThat("Метод PayCampaigns вернул неуспешный код завершения операции", response, equalTo(1));
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChangedOnFunds(CLIENT_LOGIN, campaign1, SUM_NO_VAT));
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChangedOnFunds(CLIENT_LOGIN, campaign2, SUM_NO_VAT));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6380")
    public void testSeveralPaysOneCampaign() {
        api.as(AGENCY_LOGIN);
        log.info("Оплата одной кампании дважды в одном вызове");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(354, AxisErrorDetails.PAYCAMPELEMENT_CAMPAIGNID_MUST_BE_UNIQUE),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID(defaultContract)
                        .withPayments(new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                    .withCurrency(Currency.RUB)
                                        .withSum(sum),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum)));
    }

    /**
     * попытка оплатить предоплатным договором для юрлица
     */
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6381")
    public void testPayCampaignContractPayBeforeUr() {
        api.as(AGENCY_LOGIN);
        log.info("Попытка оплаты кампании предоплатным договором (юр.)");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(355, null),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID(
                                api.userSteps.balanceSteps().getContractPayBefore(User.get(AGENCY_LOGIN).getClientID()))
                        .withPayments(new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum))
        );
    }

    /**
     * попытка использовать для оплаты просроченный контракт
     */
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6382")
    public void testPayCampaignOldContact() {
        api.as(AGENCY_LOGIN);
        log.info("Попытка оплаты кампании истекшим договором");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(356, null),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID(
                                api.userSteps.balanceSteps().getOldContract(User.get(AGENCY_LOGIN).getClientID()))
                        .withPayments(new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum))
        );
    }

    /**
     * оплата несуществующей кампании
     */
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6383")
    public void testPayNotExistCampaign() {
        api.as(AGENCY_LOGIN);
        log.info("Попытка оплаты несуществующей (была удалена) кампании");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(1, null),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID(defaultContract)
                        .withPayments(new PayCampElementMap(api.type())
                                        .withCampaignID(deletedCampaignID)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum))
        );
    }

    /**
     * оплата договором, с исчерпанным лимитом
     */
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6384")
    public void testPayCampaignContractWithSmallLimit() {
        api.as(AGENCY_LOGIN);
        log.info("Попытка оплаты кампании договором, с исчерпанным лимитом");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(355, null),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID(
                                api.userSteps.balanceSteps()
                                        .getContractWithSmallLimit(User.get(AGENCY_LOGIN).getClientID()))
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(sum))
        );
    }

    /**
     * попытка использовать для оплаты аннулированный контракт
     */
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6385")
    public void testPayCampaignContactDisabled() {
        api.as(AGENCY_LOGIN);
        log.info("Попытка оплаты кампании аннулированным договором");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(356, null),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID(
                                "32576/15") //TODO жду, когда в Балансе сделают возможность получать такой договор
                        .withPayments(new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum))
        );
    }

    /**
     * попытка оплатить не своими кредитными лимитами
     */
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6386")
    public void testPayCampaignNotMyContract() {
        api.as(AGENCY_LOGIN);
        log.info("Попытка оплаты кампании чужим договором");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(356, null),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID(api.userSteps.balanceSteps()
                                .getDefaultContract(User.get(Logins.AGENCY_ANOTHER).getClientID()))
                        .withPayments(new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum))
        );
    }

    /**
     * оплата несуществующим договором
     */
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6387")
    public void testPayCampaignContractNonExist() {
        api.as(AGENCY_LOGIN);
        log.info("Попытка оплаты кампании несуществующим договором");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(71, AxisErrorDetails.FIELD_COMPLETED_INCORRECTLY,
                        capitalize(PayCampaignsInfoMap.CONTRACT_ID)),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID("00000/00")
                        .withPayments(new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum))
        );
    }

    // https://jira.yandex-team.ru/browse/DIRECT-24803
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6388")
    public void testPayCampaignOneNotMyCampaign() {
        api.as(Logins.SUPER_LOGIN, "holodilnikru");
        log.info("Попытка оплатить кампанию, на которую нет прав  вместе с кампанией, на которую есть права (263)");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(54, null),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.BANK)
                        .withContractID(defaultContract)
                        .withPayments(new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(263)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sum))
        );
    }

//    @Test
//    public void testPayCampaignAuthOldMethod() {
//        api.as(AGENCY_LOGIN);
//        log.info("Оплата кампании по кредитному договору на сумму: " + sum);
//        CampaignInfo campaignInfo = api.userSteps.campaignSteps().getCampaignParams(activeCampaignID1);
//        int response = api.userSteps.financeSteps().payCampaigns(
//                PayMethod.BANK,
//                defaultContract,
//                true,
//                new PayCampElementMap(api.type())
//                        .withCampaignID(activeCampaignID1)
//                        .withSum(sum));
//        assertThat("Метод PayCampaigns вернул неуспешный код завершения операции", response, equalTo(1));
//        ConditionFactories.NOTIFY_ORDER2.until(
//                api.userSteps.campaignSteps().campaignSumChangedOnSum(new CampaignInfoMap(campaignInfo), sum));
//    } ToDo broken payment token - DIRECT-41294
}
