package ru.yandex.autotests.directapi.finance.paycampaigns;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.Account;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.matchers.finance.AccountMatcher;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.PayCampaignsInfoMap;
import ru.yandex.autotests.directapi.model.finances.PayMethod;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;

/**
 * User: mariabye
 * Date: 17.10.13
 */
@Aqua.Test
@Features(FinanceFeatures.PAY_CAMPAIGNS)
public class PayCampaignsOverdraftAccountTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    static String login = FinanceLogins.FREE_CLIENT_OVERDRAFT2;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static long campaignID1;
    private static long campaignID2;

    @BeforeClass
    public static void getAccountIDs() {
        campaignID1 = api.userSteps.addModeratedCampaign(login);
        api.as(Logins.SUPER_LOGIN).userSteps.financeSteps().verifySharedAccount(login);
        api.as(login);
        campaignID2 = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.balanceSteps().synchronizeWithBalance(
                ((Account) api.userSteps.financeSteps().getAccount(login)).getAccountID());
    }

    Money minPaymentAmount = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount();

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6416")
    public void payCampaign() {
        Account account = api.userSteps.financeSteps().getAccount(login);
        Float amount = account.getAmount();
        api.userSteps.financeSteps().payCampaignsOverdraft(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minPaymentAmount.addVAT().floatValue())
        );


        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.financeSteps().accountAmountChanged(account));
        account = api.userSteps.financeSteps().getAccount(login);
        assertThat("Неверная сумма на счёте", account.getAmount(), equalTo(minPaymentAmount.add(amount).floatValue()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6417")
    public void payCampaign0Amount() {
        int accountID = api.userSteps.financeSteps().getAccountID(login);
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(354, AxisErrorDetails.INVALID_AMOUNT_SPECIFIED_FOR_ACCOUNT, accountID),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.OVERDRAFT)
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(campaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(0f))
        );

    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6412")
    public void payCampaignLessAmount() {
        int accountID = api.userSteps.financeSteps().getAccountID(login);
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(354, AxisErrorDetails.LOW_CURRENCY_PAYMENT_AMOUNT_FOR_ACCOUNT,
                        MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount()
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        MoneyCurrency.get(Currency.RUB).getApiDescription(), accountID),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.OVERDRAFT)
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(campaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(minPaymentAmount.getPrevious().floatValue()))
        );

    }

    //@Test до починки ЯД DIRECT-65926
    @ru.yandex.qatools.allure.annotations.TestCaseId("6413")
    public void payCampaignWithPaymentToken() {
        Account account = api.userSteps.financeSteps().getAccount(login);
        Float amount = account.getAmount();
        api.userSteps.financeSteps().payCampaigns(
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.OVERDRAFT)
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(campaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(minPaymentAmount.addVAT().floatValue())),
                true
        );
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.financeSteps().accountAmountChanged(account));
        account = api.userSteps.financeSteps().getAccount(login);
        assertThat(account, AccountMatcher.hasAmount(equalTo(minPaymentAmount.add(amount).floatValue())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6410")
    public void payCampaigns() {
        Account account = api.userSteps.financeSteps().getAccount(login);
        Float amount = account.getAmount();
        api.userSteps.financeSteps().payCampaignsOverdraft(
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID1)
                        .withCurrency(Currency.RUB)
                        .withSum(minPaymentAmount.addVAT().floatValue()),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignID2)
                        .withCurrency(Currency.RUB)
                        .withSum(minPaymentAmount.addVAT().floatValue())
        );


        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.financeSteps().accountAmountChangedOnSum(
                        account, minPaymentAmount.multiply(2f).floatValue()), equalTo(true));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6411")
    public void payAccountID() {
        Account account = api.userSteps.financeSteps().getAccount(login);
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaigns(
                new AxisError(1),
                new PayCampaignsInfoMap(api.type())
                        .withPayMethod(PayMethod.OVERDRAFT)
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(account.getAccountID())
                                .withCurrency(Currency.RUB)
                                .withSum(minPaymentAmount.addVAT().floatValue()))
        );
    }
}
