package ru.yandex.autotests.directapi.finance.paycampaignsbycard;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.PayCampaignsByCardInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;

/**
 * Created by semkagtn on 16.09.15.
 * https://st.yandex-team.ru/TESTIRT-6408
 */
@Aqua.Test
@Features(FinanceFeatures.PAY_CAMPAIGNS_BY_CARD)
@Issue("https://st.yandex-team.ru/DIRECT-42633")
@Description("Вызов метода для кампаний в разных статусах")
public class PayCampaignsByCardCampaignsDifferentStatusesTest {

    private static final String LOGIN = Logins.CLIENT_FOR_RUB;
    private static final Money TRANSFER_AMOUNT = MoneyCurrency.get(Currency.RUB).getMinTransferAmount().addVAT();

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(LOGIN);

    @Rule
    public Trashman trasher = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static String customTransactionId = randomAlphanumeric(
            PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID_LENGTH);

    private long campaignId;

    @Before
    @Step("Подготовка данных для теста")
    public void createCampaign() {
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
    }

    @Test
    @Description("Вызов метода для активной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("6461")
    public void requestForActiveCampaign() {
        api.userSteps.campaignFakeSteps().makeCampaignActive(campaignId);

        api.userSteps.financeSteps().shouldGetErrorOn(Method.PAY_CAMPAIGNS_BY_CARD,
                new PayCampaignsByCardInfoMap(api.type())
                        .withPayMethodID("someInvalidPaymentMethod")
                        .withCustomTransactionID(customTransactionId)
                        .withVersion("20502A")
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(campaignId)
                                .withCurrency(Currency.RUB)
                                .withSum(TRANSFER_AMOUNT)),
                new AxisError(368, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @Description("Вызов метода для активной и промодерированной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("6462")
    public void requestForActiveAndModeratedCampaign() {
        api.userSteps.campaignFakeSteps().makeCampaignActive(campaignId);
        long anotherCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().makeCampaignFullyModerated(anotherCampaignId);

        api.userSteps.financeSteps().shouldGetErrorOn(Method.PAY_CAMPAIGNS_BY_CARD,
                new PayCampaignsByCardInfoMap(api.type())
                        .withPayMethodID("someInvalidPaymentMethodLol")
                        .withCustomTransactionID(customTransactionId)
                        .withVersion("20502A")
                        .withPayments(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignId)
                                        .withCurrency(Currency.RUB)
                                        .withSum(TRANSFER_AMOUNT),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(anotherCampaignId)
                                        .withCurrency(Currency.RUB)
                                        .withSum(TRANSFER_AMOUNT)),
                new AxisError(368, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @Description("Вызов метода для кампании-черновика")
    @ru.yandex.qatools.allure.annotations.TestCaseId("6463")
    public void requestForDraftCampaign() {
        api.userSteps.financeSteps().shouldGetErrorOn(Method.PAY_CAMPAIGNS_BY_CARD,
                new PayCampaignsByCardInfoMap(api.type())
                        .withPayMethodID("someInvalidPaymentMethod")
                        .withCustomTransactionID(customTransactionId)
                        .withVersion("20502A")
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(campaignId)
                                .withCurrency(Currency.RUB)
                                .withSum(TRANSFER_AMOUNT)),
                new AxisError(354, AxisErrorDetails.CAMPAIGN_NOT_MODERATE, campaignId));
    }

    @Test
    @Description("Вызов метода для остановленой непромодерированной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("6464")
    public void requestForStoppedNotModeratedCampaign() {
        api.userSteps.campaignSteps().campaignsSuspend(campaignId);

        api.userSteps.financeSteps().shouldGetErrorOn(Method.PAY_CAMPAIGNS_BY_CARD,
                new PayCampaignsByCardInfoMap(api.type())
                        .withPayMethodID("someInvalidPaymentMethod")
                        .withCustomTransactionID(customTransactionId)
                        .withVersion("20502A")
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(campaignId)
                                .withCurrency(Currency.RUB)
                                .withSum(TRANSFER_AMOUNT)),
                new AxisError(354, AxisErrorDetails.CAMPAIGN_NOT_MODERATE, campaignId));
    }

    @Test
    @Description("Вызов метода для заархивированной непромодерированой кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("6465")
    public void requestForArchivedNotModeratedCampaign() {
        api.userSteps.campaignSteps().campaignsSuspend(campaignId);
        api.userSteps.campaignSteps().campaignsArchive(campaignId);

        api.userSteps.financeSteps().shouldGetErrorOn(Method.PAY_CAMPAIGNS_BY_CARD,
                new PayCampaignsByCardInfoMap(api.type())
                        .withPayMethodID("someInvalidPaymentMethod")
                        .withCustomTransactionID(customTransactionId)
                        .withVersion("20502A")
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(campaignId)
                                .withCurrency(Currency.RUB)
                                .withSum(TRANSFER_AMOUNT)),
                new AxisError(354, AxisErrorDetails.CAMPAIGN_NOT_MODERATE, campaignId));
    }

    @Test
    @Description("Вызов метода для остановленой промодерированной кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("6466")
    public void requestForStoppedModeratedCampaign() {
        api.userSteps.campaignFakeSteps().makeCampaignFullyModerated(campaignId);
        api.userSteps.campaignSteps().campaignsSuspend(campaignId);

        api.userSteps.financeSteps().shouldGetErrorOn(Method.PAY_CAMPAIGNS_BY_CARD,
                new PayCampaignsByCardInfoMap(api.type())
                        .withPayMethodID("someInvalidPaymentMethod")
                        .withCustomTransactionID(customTransactionId)
                        .withVersion("20502A")
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(campaignId)
                                .withCurrency(Currency.RUB)
                                .withSum(TRANSFER_AMOUNT)),
                new AxisError(368));
    }

    @Test
    @Description("Вызов метода для заархивированной промодерированой кампании")
    @ru.yandex.qatools.allure.annotations.TestCaseId("6467")
    public void requestForArchivedModeratedCampaign() {
        api.userSteps.campaignFakeSteps().makeCampaignFullyModerated(campaignId);
        api.userSteps.campaignSteps().campaignsSuspend(campaignId);
        api.userSteps.campaignSteps().campaignsArchive(campaignId);

        api.userSteps.financeSteps().shouldGetErrorOn(Method.PAY_CAMPAIGNS_BY_CARD,
                new PayCampaignsByCardInfoMap(api.type())
                        .withPayMethodID("someInvalidPaymentMethod")
                        .withCustomTransactionID(customTransactionId)
                        .withVersion("20502A")
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(campaignId)
                                .withCurrency(Currency.RUB)
                                .withSum(TRANSFER_AMOUNT)),
                new AxisError(368, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @Description("Вызов метода для промодерированной кампании и кампании черновика")
    @ru.yandex.qatools.allure.annotations.TestCaseId("6468")
    public void requestForModeratedAndDraftCampaign() {
        api.userSteps.campaignFakeSteps().makeCampaignFullyModerated(campaignId);
        long anotherCampaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();

        api.userSteps.financeSteps().shouldGetErrorOn(Method.PAY_CAMPAIGNS_BY_CARD,
                new PayCampaignsByCardInfoMap(api.type())
                        .withPayMethodID("someInvalidPaymentMethodLol")
                        .withCustomTransactionID(customTransactionId)
                        .withVersion("20502A")
                        .withPayments(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignId)
                                        .withCurrency(Currency.RUB)
                                        .withSum(TRANSFER_AMOUNT),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(anotherCampaignId)
                                        .withCurrency(Currency.RUB)
                                        .withSum(TRANSFER_AMOUNT)),
                new AxisError(354, AxisErrorDetails.CAMPAIGN_NOT_MODERATE, anotherCampaignId));
    }
}
