package ru.yandex.autotests.directapi.finance.paycampaignsbycard;

import java.util.Arrays;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.PayCampaignsByCardInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.allure.annotations.Step;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;

/**
 * Created by semkagtn on 16.09.15.
 * https://st.yandex-team.ru/TESTIRT-6408
 */
@Aqua.Test
@Features(FinanceFeatures.PAY_CAMPAIGNS_BY_CARD)
@Issue("https://st.yandex-team.ru/DIRECT-42633")
@Description("Проверка минимальной возможной суммы в запросе для каждой валюты")
@RunWith(Parameterized.class)
public class PayCampaignsByCardSumMinimumValueTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @Rule
    public Trashman trasher = new Trashman(api);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Parameterized.Parameter(0)
    public Currency currency;

    @Parameterized.Parameter(1)
    public String login;

    @Parameterized.Parameter(2)
    public String stringCurrency;

    @Parameterized.Parameters(name = "Currency = {0}")
    public static Iterable<Object[]> data() {
        return Arrays.asList(new Object[][]{
                {Currency.RUB, Logins.LOGIN_FOR_RUB,
                        MoneyCurrency.get(Currency.RUB).getAPIAbbreviation(api.locale())},
        });
    }

    private static String customTransactionId = randomAlphanumeric(
            PayCampaignsByCardInfoMap.CUSTOM_TRANSACTION_ID_LENGTH);

    private long campaignId;

    @Before
    @Step("Подготовка данных для теста")
    public void createCampaign() {
        api.as(login);
        campaignId = api.userSteps.campaignSteps().addDefaultTextCampaign();
        api.userSteps.campaignFakeSteps().makeCampaignFullyModerated(campaignId);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6487")
    public void lessThanMinimumSumInRequest() {
        Money minSum = MoneyCurrency.get(currency).getMinInvoiceAmount();
        String minSumString = minSum.stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED);

        AxisError expectedError = currency == Currency.YND_FIXED
                ? new AxisError(354, AxisErrorDetails.LOW_CURRENCY_PAYMENT_AMOUNT_WITHOUT_VAT,
                minSumString, stringCurrency, campaignId)
                : new AxisError(354, AxisErrorDetails.LOW_CURRENCY_PAYMENT_AMOUNT,
                        minSumString, stringCurrency, campaignId);

        api.userSteps.financeSteps().shouldGetErrorOn(Method.PAY_CAMPAIGNS_BY_CARD,
                new PayCampaignsByCardInfoMap(api.type())
                        .withPayMethodID("someInvalidPaymentMethod")
                        .withCustomTransactionID(customTransactionId)
                        .withVersion("20502A")
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(campaignId)
                                .withCurrency(currency)
                                .withSum(minSum.addVAT().getPrevious().floatValue())),
                expectedError);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6486")
    public void minSumInRequest() {
        float sum = MoneyCurrency.get(currency).getMinInvoiceAmount().addVAT().floatValue();

        api.userSteps.financeSteps().shouldGetErrorOn(Method.PAY_CAMPAIGNS_BY_CARD,
                new PayCampaignsByCardInfoMap(api.type())
                        .withPayMethodID("someInvalidPaymentMethodLOL")
                        .withCustomTransactionID(customTransactionId)
                        .withVersion("20502A")
                        .withPayments(new PayCampElementMap(api.type())
                                .withCampaignID(campaignId)
                                .withCurrency(currency)
                                .withSum(sum)),
                new AxisError(368, AxisErrorDetails.EMPTY_STRING));
    }
}
