package ru.yandex.autotests.directapi.finance.transfermoney;

import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.aqua.annotations.project.Feature;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.TransferMoneyInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;

/**
 * Author pavryabov
 * Date 26.05.14
 * https://jira.yandex-team.ru/browse/TESTIRT-513
 */
@Aqua.Test
@Feature(FinanceFeatures.TRANSFER_MONEY)
@Stories(ApiStories.CURRENCY)
@RunWith(Parameterized.class)
public class RequestCurrencyTest {
    protected LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private long campaignIDFrom;
    private long campaignIDTo;

    private Money startSum = Money.valueOf(2000);

    @Parameterized.Parameter
    public String client;

    @Parameterized.Parameter(1)
    public Currency currency;

    @Parameterized.Parameters(name = "Client = {0}")
    public static Collection logins() {
        Object[][] data = new Object[][]{
// DIRECT-66511                {Logins.LOGIN_FOR_UAH, Currency.UAH},
                {Logins.LOGIN_FOR_KZT, Currency.KZT}, // DIRECT-65042
                {Logins.LOGIN_FOR_EUR, Currency.EUR},
                {Logins.LOGIN_FOR_CHF, Currency.CHF},
// DIRECT-130518    {Logins.LOGIN_FOR_TRY, Currency.TRY}
        };
        return Arrays.asList(data);
    }

    @Before
    public void initTest() {
        api.as(client);
        startSum = startSum.convert(currency);
        campaignIDFrom = api.userSteps.addActiveCampaign(startSum.floatValue(), currency);
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps().campaignFundsChanged(campaignIDFrom));
        campaignIDTo = api.userSteps.addModeratedCampaign();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6511")
    public void callTransferMoneyWithValidCurrency() {
        Money sum = MoneyCurrency.get(currency).getMinTransferAmount();
        api.userSteps.financeSteps().transferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDFrom)
                                        .withSum(sum)
                                        .withCurrency(currency)
                        )
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDTo)
                                        .withSum(sum)
                                        .withCurrency(currency)
                        )
        );
        Money transferSumInBalance = sum.addVAT();
        Money expectedSumFrom = startSum.subtract(transferSumInBalance).subtractVAT()
                .setScale(2, RoundingMode.HALF_UP);
        Money expectedSumTo = transferSumInBalance.subtractVAT().setScale(2, RoundingMode.HALF_UP);
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsSumIs(campaignIDFrom), equalTo(expectedSumFrom));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsSumIs(campaignIDTo), equalTo(expectedSumTo));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6512")
    public void expectErrorTransferMoneyWithInvalidCurrency() {
        float sumFloat = MoneyCurrency.get(Currency.RUB).getMinTransferAmount().floatValue();
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(campaignIDFrom, campaignIDTo, sumFloat, Currency.RUB,
                new AxisError(245, AxisErrorDetails.VALID_CURRENCY_VALUE_IS, currency));
    }
}
