package ru.yandex.autotests.directapi.finance.transfermoney;

import java.math.RoundingMode;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.aqua.annotations.project.Feature;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.TransferMoneyInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * Author pavryabov
 * Date 25.05.14
 * https://jira.yandex-team.ru/browse/TESTIRT-513
 */
@Aqua.Test
@Feature(FinanceFeatures.TRANSFER_MONEY)
@Stories(ApiStories.CURRENCY)
@Tag(TagDictionary.TRUNK)
public class TransferMoneyCurrencyNegativeTest {
    protected LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static String rubClient = Logins.LOGIN_FOR_RUB;
    private static String usdClient = Logins.LOGIN_FOR_USD;

    private static long rubCampaignID1;
    private static long rubCampaignID2;
    private static long rubCampaignID3;
    private static long usdCampaignID1;
    private static long usdCampaignID2;
    private static long usdCampaignID3;

    @BeforeClass
    public static void createCampaigns() {
        api.as(rubClient);
        rubCampaignID1 = api.userSteps.addActiveCampaign(rubClient, 1);
        rubCampaignID2 = api.userSteps.addActiveCampaign(rubClient, 1);
        rubCampaignID3 = api.userSteps.addActiveCampaign(rubClient, 1);
        api.as(usdClient);
        usdCampaignID1 = api.userSteps.addActiveCampaign(usdClient, 1);
        usdCampaignID2 = api.userSteps.addActiveCampaign(usdClient, 1);
        usdCampaignID3 = api.userSteps.addActiveCampaign(usdClient, 1);
    }

    //Перевод с одной кампании на одну кампанию. Рубли
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6530")
    public void transferMoneyFromOneRUBSumToOneRUBSum() {
        api.as(rubClient);
        log.info("Вызываем метод TransferMoney для кампаний рублевого клиента с параметром Currency = RUB " +
                "в FromCampaigns и ToCampaigns. Выполняется перевод 600 -> 599.99");
        Money sumFrom = Money.valueOf(600f);
        Money sumTo = Money.valueOf(599.99f);
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(rubCampaignID1)
                                        .withSum(sumFrom.floatValue())
                                        .withCurrency(Currency.RUB)
                        )
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(rubCampaignID2)
                                        .withSum(sumTo.floatValue())
                                        .withCurrency(Currency.RUB)
                        ),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_AMOUNT_NOT_EQUAL_RUS,
                        sumFrom.setScale(2, RoundingMode.HALF_UP).stringValue(MoneyFormat.TWO_DIGITS_POINT),
                        sumTo.setScale(2, RoundingMode.HALF_UP).stringValue(MoneyFormat.TWO_DIGITS_POINT)));
    }

    //Перевод с двух кампаний на одну кампанию. Рубли.
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6532")
    public void transferMoneyFromTwoRUBSumToOneRUBSum() {
        api.as(rubClient);
        log.info("Вызываем метод TransferMoney для кампаний рублевого клиента с параметром Currency = RUB " +
                "в FromCampaigns и ToCampaigns. Выполняется перевод 300.17 + 300.27 -> 600.43");
        Money sumFrom1 = Money.valueOf(300.17f);
        Money sumFrom2 = Money.valueOf(300.27f);
        Money sumTo = Money.valueOf(600.43f);
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(rubCampaignID1)
                                        .withSum(sumFrom1.floatValue())
                                        .withCurrency(Currency.RUB),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(rubCampaignID2)
                                        .withSum(sumFrom2.floatValue())
                                        .withCurrency(Currency.RUB)
                        )
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(rubCampaignID3)
                                        .withSum(sumTo.floatValue())
                                        .withCurrency(Currency.RUB)
                        ),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_AMOUNT_NOT_EQUAL_RUS,
                        sumFrom1.setScale(2, RoundingMode.HALF_UP).add(sumFrom2.setScale(2, RoundingMode.HALF_UP))
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT),
                        sumTo.setScale(2, RoundingMode.HALF_UP).stringValue(MoneyFormat.TWO_DIGITS_POINT)));
    }

    //Перевод с одной кампании на две кампании. Рубли.
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6534")
    public void transferMoneyFromOneRUBSumToTwoRUBSum() {
        api.as(rubClient);
        log.info("Вызываем метод TransferMoney для кампаний рублевого клиента с параметром Currency = RUB " +
                "в FromCampaigns и ToCampaigns. Выполняется перевод  600.53 -> 300.27 + 300.27");
        Money sumFrom = Money.valueOf(600.53f);
        Money sumTo1 = Money.valueOf(300.27f);
        Money sumTo2 = Money.valueOf(300.27f);
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(rubCampaignID1)
                                        .withSum(sumFrom.floatValue())
                                        .withCurrency(Currency.RUB)
                        )
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(rubCampaignID2)
                                        .withSum(sumTo1.floatValue())
                                        .withCurrency(Currency.RUB),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(rubCampaignID3)
                                        .withSum(sumTo2.floatValue())
                                        .withCurrency(Currency.RUB)
                        ),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_AMOUNT_NOT_EQUAL_RUS,
                        sumFrom.setScale(2, RoundingMode.HALF_UP).stringValue(MoneyFormat.TWO_DIGITS_POINT),
                        sumTo1.setScale(2, RoundingMode.HALF_UP).add(sumTo2.setScale(2, RoundingMode.HALF_UP))
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT)));
    }

    //Перевод с одной кампании на одну кампанию. Доллары
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6536")
    public void transferMoneyFromOneUSDSumToOneUSDSum() {
        api.as(usdClient);
        log.info("Вызываем метод TransferMoney для кампаний долларового клиента с параметром Currency = USD " +
                "в FromCampaigns и ToCampaigns. Выполняется перевод 20.01 -> 20");
        Money sumFrom = Money.valueOf(20.01f);
        Money sumTo = Money.valueOf(20f);
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(usdCampaignID1)
                                        .withSum(sumFrom.floatValue())
                                        .withCurrency(Currency.USD)
                        )
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(usdCampaignID2)
                                        .withSum(sumTo.floatValue())
                                        .withCurrency(Currency.USD)
                        ),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_AMOUNT_NOT_EQUAL_RUS,
                        sumFrom.setScale(2, RoundingMode.HALF_UP).stringValue(MoneyFormat.TWO_DIGITS_POINT),
                        sumTo.setScale(2, RoundingMode.HALF_UP).stringValue(MoneyFormat.TWO_DIGITS_POINT)));
    }

    //Перевод с двух кампаний на одну кампанию. Доллары
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6538")
    public void transferMoneyFromTwoUSDSumToOneUSDSum() {
        api.as(usdClient);
        log.info("Вызываем метод TransferMoney для кампаний долларового клиента с параметром Currency = USD " +
                "в FromCampaigns и ToCampaigns. Выполняется перевод 10.999 + 10.999 -> 21.989");
        Money sumFrom1 = Money.valueOf(10.999f);
        Money sumFrom2 = Money.valueOf(10.999f);
        Money sumTo = Money.valueOf(21.989f);
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(usdCampaignID1)
                                        .withSum(sumFrom1.floatValue())
                                        .withCurrency(Currency.USD),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(usdCampaignID2)
                                        .withSum(sumFrom2.floatValue())
                                        .withCurrency(Currency.USD)
                        )
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(usdCampaignID3)
                                        .withSum(sumTo.floatValue())
                                        .withCurrency(Currency.USD)
                        ),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_AMOUNT_NOT_EQUAL_RUS,
                        sumFrom1.setScale(2, RoundingMode.HALF_UP).add(sumFrom2.setScale(2, RoundingMode.HALF_UP))
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT),
                        sumTo.setScale(2, RoundingMode.HALF_UP).stringValue(MoneyFormat.TWO_DIGITS_POINT)));
    }

    //Перевод с одной кампании на две кампании. Доллары
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6540")
    public void transferMoneyFromOneUSDSumToTwoUSDSum() {
        api.as(usdClient);
        log.info("Вызываем метод TransferMoney для кампаний долларового клиента с параметром Currency = USD " +
                "в FromCampaigns и ToCampaigns. Выполняется перевод  21.989 ->10.999 + 10.999");
        Money sumFrom = Money.valueOf(21.989f);
        Money sumTo1 = Money.valueOf(10.999f);
        Money sumTo2 = Money.valueOf(10.999f);
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(usdCampaignID1)
                                        .withSum(sumFrom.floatValue())
                                        .withCurrency(Currency.USD)
                        )
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(usdCampaignID2)
                                        .withSum(sumTo1.floatValue())
                                        .withCurrency(Currency.USD),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(usdCampaignID3)
                                        .withSum(sumTo2.floatValue())
                                        .withCurrency(Currency.USD)
                        ),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_AMOUNT_NOT_EQUAL_RUS,
                        sumFrom.setScale(2, RoundingMode.HALF_UP).stringValue(MoneyFormat.TWO_DIGITS_POINT),
                        sumTo1.setScale(2, RoundingMode.HALF_UP).add(sumTo2.setScale(2, RoundingMode.HALF_UP))
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT)));
    }
}
