package ru.yandex.autotests.directapi.finance.transfermoney;

import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.aqua.annotations.project.Feature;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.common.api45.ClientInfo;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.TransferMoneyInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Author pavryabov
 * Date 25.05.14
 * https://jira.yandex-team.ru/browse/TESTIRT-51
 */
@Aqua.Test
@Feature(FinanceFeatures.TRANSFER_MONEY)
@Stories(ApiStories.CURRENCY)
@RunWith(Parameterized.class)
public class TransferMoneyCurrencyOneToTwoCampaignsTest {
    protected LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static String rubClient = Logins.LOGIN_FOR_RUB;
    private static String usdClient = Logins.LOGIN_FOR_USD;
    private static String bynClient = Logins.BYN_CLIENT2;
    private static String discountClient = FinanceLogins.DISCOUNT_CLIENT;
    private long campaignIDFrom;
    private long campaignIDTo1;
    private long campaignIDTo2;

    private Money startSum = Money.valueOf(900f);

    @Parameterized.Parameter(0)
    public String client;

    @Parameterized.Parameter(1)
    public Currency clientCurrency;

    @Parameterized.Parameter(2)
    public Currency transferCurrency;

    @Parameterized.Parameter(3)
    public Float sumFrom;

    @Parameterized.Parameter(4)
    public Float sumTo1;

    @Parameterized.Parameter(5)
    public Float sumTo2;

    @Parameterized.Parameter(6)
    public Float discount;

    @Parameterized.Parameters(name = "Client = {0}, TransferCurrency = {2}")
    public static Collection logins() {
        Object[][] data = new Object[][]{
                {rubClient, Currency.RUB, Currency.RUB, 2000.54f, 1000.27f, 1000.27f, 0f},
                {usdClient, Currency.USD, Currency.USD, 41.989f, 20.999f, 20.989f, 0f},
                {bynClient, Currency.BYN, Currency.BYN, 81.989f, 40.999f, 40.989f, 0f},
                {discountClient, Currency.RUB, Currency.RUB, 2000.54f, 1000.27f, 1000.27f, 5f}
        };
        return Arrays.asList(data);
    }

    @Before
    public void initTest() {
        api.as(client);
        if (discount > 0f) {
            api.userSteps.clientFakeSteps().setDiscount(User.get(client).getClientID(), discount.intValue());
        }
        assumeThat("у клиента правильная скидка",
                ((ClientInfo) api.userSteps.clientSteps().getClientInfo(client)).getDiscount(), equalTo(discount));
        discount = (1 - discount / 100);
        startSum = startSum.convert(clientCurrency);
        campaignIDFrom = api.userSteps.addActiveCampaign(startSum.floatValue(), clientCurrency);
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps().campaignFundsChanged(campaignIDFrom));
        campaignIDTo1 = api.userSteps.addModeratedCampaign();
        campaignIDTo2 = api.userSteps.addModeratedCampaign();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6543")
    public void transferMoneyFromOneRUBSumToOneRUBSum() {
        api.userSteps.financeSteps().transferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDFrom)
                                        .withSum(sumFrom)
                                        .withCurrency(transferCurrency)
                        )
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDTo1)
                                        .withSum(sumTo1)
                                        .withCurrency(transferCurrency),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDTo2)
                                        .withSum(sumTo2)
                                        .withCurrency(transferCurrency)
                        )
        );
        if (transferCurrency == null) {
            sumFrom = sumFrom * discount;
            sumTo1 = sumTo1 * discount;
            sumTo2 = sumTo2 * discount;
        }
        Money transferSumInBalanceFrom =
                Money.valueOf(sumFrom, transferCurrency).convert(clientCurrency)
                        .setScale(2, RoundingMode.HALF_UP).addVAT().setScale(6, RoundingMode.HALF_UP);
        Money expectedSumFrom =
                startSum.subtract(transferSumInBalanceFrom).subtractVAT().setScale(2, RoundingMode.HALF_UP);
        Money transferSumInBalanceTo1 =
                Money.valueOf(sumTo1, transferCurrency).convert(clientCurrency)
                        .setScale(2, RoundingMode.HALF_UP).addVAT().setScale(6, RoundingMode.HALF_UP);
        Money transferSumInBalanceTo2 =
                Money.valueOf(sumTo2, transferCurrency).convert(clientCurrency)
                        .setScale(2, RoundingMode.HALF_UP).addVAT().setScale(6, RoundingMode.HALF_UP);
        Money expectedSumTo1 = transferSumInBalanceFrom.multiply(transferSumInBalanceTo1.floatValue())
                .divide(transferSumInBalanceTo1.add(transferSumInBalanceTo2).floatValue())
                .setScale(6, RoundingMode.HALF_UP).subtractVAT().setScale(2, RoundingMode.HALF_UP);
        Money expectedSumTo2 = transferSumInBalanceFrom.multiply(transferSumInBalanceTo2.floatValue())
                .divide(transferSumInBalanceTo1.add(transferSumInBalanceTo2).floatValue())
                .setScale(6, RoundingMode.HALF_UP).subtractVAT().setScale(2, RoundingMode.HALF_UP);
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsSumIs(campaignIDFrom), equalTo(expectedSumFrom));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsSumIs(campaignIDTo1), equalTo(expectedSumTo1));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsSumIs(campaignIDTo2), equalTo(expectedSumTo2));
    }
}
