package ru.yandex.autotests.directapi.finance.transfermoney;

import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.aqua.annotations.project.Feature;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.common.api45.ClientInfo;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.finance.FinanceLogins;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.TransferMoneyInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Author pavryabov
 * Date 25.05.14
 * https://jira.yandex-team.ru/browse/TESTIRT-513
 */
@Aqua.Test
@Feature(FinanceFeatures.TRANSFER_MONEY)
@Stories(ApiStories.CURRENCY)
@RunWith(Parameterized.class)
public class TransferMoneyCurrencyTwoToOneCampaignsTest {
    protected LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static String rubClient = Logins.LOGIN_FOR_RUB;
    private static String usdClient = Logins.LOGIN_FOR_USD;
    private static String bynClient = Logins.BYN_CLIENT2;
    private static String discountClient = FinanceLogins.DISCOUNT_CLIENT;
    private long campaignIDFrom1;
    private long campaignIDFrom2;
    private long campaignIDTo;

    private Money startSum = Money.valueOf(1000f);

    @Parameterized.Parameter(0)
    public String client;

    @Parameterized.Parameter(1)
    public Currency clientCurrency;

    @Parameterized.Parameter(2)
    public Currency transferCurrency;

    @Parameterized.Parameter(3)
    public Float sumFrom1;

    @Parameterized.Parameter(4)
    public Float sumFrom2;

    @Parameterized.Parameter(5)
    public Float sumTo;

    @Parameterized.Parameter(6)
    public Float discount;

    @Parameterized.Parameters(name = "Client = {0}, TransferCurrency = {2}")
    public static Collection logins() {
        Object[][] data = new Object[][]{
                {rubClient, Currency.RUB, Currency.RUB, 1000.27f, 1000.27f, 2000.54f, 0f},
                {usdClient, Currency.USD, Currency.USD, 20.989f, 20.999f, 41.989f, 0f},
                {bynClient, Currency.BYN, Currency.BYN, 60.989f, 60.999f, 121.989f, 0f},
                {discountClient, Currency.RUB, Currency.RUB, 1000.27f, 1000.27f, 2000.54f, 5f}
        };
        return Arrays.asList(data);
    }

    @Before
    public void initTest() {
        api.as(client);
        if (discount > 0f) {
            api.userSteps.clientFakeSteps().setDiscount(User.get(client).getClientID(), discount.intValue());
        }
        assumeThat("у клиента правильная скидка",
                ((ClientInfo) api.userSteps.clientSteps().getClientInfo(client)).getDiscount(), equalTo(discount));
        discount = (1 - discount / 100);
        startSum = startSum.convert(clientCurrency);
        campaignIDFrom1 = api.userSteps.addActiveCampaign(startSum.floatValue(), clientCurrency);
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps().campaignFundsChanged(campaignIDFrom1));
        campaignIDFrom2 = api.userSteps.addActiveCampaign(startSum.floatValue(), clientCurrency);
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps().campaignFundsChanged(campaignIDFrom2));
        campaignIDTo = api.userSteps.addModeratedCampaign();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6544")
    public void transferMoneyFromOneRUBSumToOneRUBSum() {
        api.userSteps.financeSteps().transferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDFrom1)
                                        .withSum(sumFrom1)
                                        .withCurrency(transferCurrency),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDFrom2)
                                        .withSum(sumFrom2)
                                        .withCurrency(transferCurrency)
                        )
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDTo)
                                        .withSum(sumTo)
                                        .withCurrency(transferCurrency)
                        )
        );
        if (transferCurrency == null) {
            sumFrom1 = sumFrom1 * discount;
            sumFrom2 = sumFrom2 * discount;
        }
        Money transferSumInBalance1 =
                Money.valueOf(sumFrom1, transferCurrency).convert(clientCurrency)
                        .setScale(2, RoundingMode.HALF_UP).addVAT().setScale(6, RoundingMode.HALF_UP);
        Money expectedSumFrom1 =
                startSum.subtract(transferSumInBalance1).subtractVAT().setScale(2, RoundingMode.HALF_UP);
        Money transferSumInBalance2 =
                Money.valueOf(sumFrom2, transferCurrency).convert(clientCurrency)
                        .setScale(2, RoundingMode.HALF_UP).addVAT().setScale(6, RoundingMode.HALF_UP);
        Money expectedSumFrom2 =
                startSum.subtract(transferSumInBalance2).subtractVAT().setScale(2, RoundingMode.HALF_UP);
        Money expectedSumTo = transferSumInBalance1.add(transferSumInBalance2).subtractVAT()
                .setScale(2, RoundingMode.HALF_UP);
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsSumIs(campaignIDFrom1), equalTo(expectedSumFrom1));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsSumIs(campaignIDFrom2), equalTo(expectedSumFrom2));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsSumIs(campaignIDTo), equalTo(expectedSumTo));
    }
}
