package ru.yandex.autotests.directapi.finance.transfermoney;

import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import org.json.simple.parser.JSONParser;
import org.json.simple.parser.ParseException;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.TransferMoneyInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.core.IsEqual.equalTo;

/**
 * Author pavryabov
 * Date 24.04.14
 */
@Aqua.Test
@Features(FinanceFeatures.TRANSFER_MONEY)
public class TransferMoneyLimitedPaymentJsonTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static String clientLogin = Logins.SUBCLIENT_RUB;
    private static String agencyLogin = Logins.AGENCY_RUB;
    private static Money sum = MoneyCurrency.get(Currency.RUB).getMinTransferAmount().addVAT();

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(agencyLogin).protocol(ProtocolType.JSON);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static long activeCampaignID1;
    private static long activeCampaignID2;
    private static long activeCampaignID3;

    @BeforeClass
    public static void createCampaigns() {
        activeCampaignID1 = api.userSteps.addActiveCampaign(clientLogin, sum.multiply(10).floatValue(), Currency.RUB);
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(clientLogin, activeCampaignID1));
        activeCampaignID2 = api.userSteps.addActiveCampaign(clientLogin, sum.multiply(10).floatValue(), Currency.RUB);
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(clientLogin, activeCampaignID2));
        activeCampaignID3 = api.userSteps.addActiveCampaign(clientLogin, sum.multiply(10).floatValue(), Currency.RUB);
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(clientLogin, activeCampaignID3));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6558")
    public void transferPositive() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);

        api.userSteps.financeSteps().transferMoney(activeCampaignID1, activeCampaignID2, sum.floatValue(),
                Currency.RUB);

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignFrom, sum.negate()));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignTo, sum));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6559")
    public void transferFromOnePositiveOneNegative() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignFromElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);

        MoneyCurrency rubCurrency = MoneyCurrency.get(Currency.RUB);

        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(2000f),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(280f))
                        .withToCampaigns(new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID3)
                                .withCurrency(Currency.RUB)
                                .withSum(2280f)),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_LOW_AMOUNT_WITH_PLACEHOLDERS,
                        rubCurrency.getMinTransferAmount().stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        rubCurrency.getAPIAbbreviation("en")));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFromElse),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6560")
    public void transferToOnePositiveOneNegative() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignToElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);

        MoneyCurrency rubCurrency = MoneyCurrency.get(Currency.RUB);

        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(2280f))
                        .withToCampaigns(new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(280f),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID3)
                                        .withCurrency(Currency.RUB)
                                        .withSum(2000f)),
                new AxisError(353, AxisErrorDetails.LOW_TRANSFER_AMOUNT_WITH_PLACEHOLDERS, activeCampaignID2,
                        rubCurrency.getMinTransferAmount().stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        rubCurrency.getAPIAbbreviation("en")));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignToElse),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6561")
    public void transferToSame() {
        api.userSteps.financeSteps()
                .shouldGetErrorOnTransferMoney(activeCampaignID1, activeCampaignID1, sum.floatValue(), Currency.RUB,
                        new AxisError(353, AxisErrorDetails.TRANSFERMONEY_INVALID_TARGET_CAMPAIGN, activeCampaignID1));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6562")
    public void transferNULL() throws ParseException {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);

        String request = new TransferMoneyInfoMap(api.type())
                .withFromCampaigns(new PayCampElementMap(api.type())
                        .withCampaignID(activeCampaignID1)
                        .withSum(123f))
                .withToCampaigns(new PayCampElementMap(api.type())
                        .withCampaignID(activeCampaignID2)
                        .withSum(123f)).toString();
        request = request.replaceAll("123\\.0", "null");
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(new JSONParser().parse(request),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_NOT_ALL_FIELDS_SPECIFIED));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6563")
    public void transferNegative() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);

        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID1)
                                .withSum(-10f))
                        .withToCampaigns(new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID2)
                                .withSum(-10f)),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_SUM_NEGATIVE));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6564")
    public void transferFew() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);

        MoneyCurrency rubCurrency = MoneyCurrency.get(Currency.RUB);

        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(0.01f))
                        .withToCampaigns(new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID2)
                                .withCurrency(Currency.RUB)
                                .withSum(0.01f)),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_LOW_AMOUNT_WITH_PLACEHOLDERS,
                        rubCurrency.getMinTransferAmount().stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        rubCurrency.getAPIAbbreviation("en")));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6565")
    public void transferEmpty() throws ParseException {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);

        String request = new TransferMoneyInfoMap(api.type())
                .withFromCampaigns(new PayCampElementMap(api.type())
                        .withCampaignID(activeCampaignID1)
                        .withSum(123f))
                .withToCampaigns(new PayCampElementMap(api.type())
                        .withCampaignID(activeCampaignID2)
                        .withSum(123f)).toString();
        request = request.replaceAll("123\\.0", "\"\"");
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(new JSONParser().parse(request),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_NOT_ALL_FIELDS_SPECIFIED));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6566")
    public void transferArray() throws ParseException {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);

        String request = new TransferMoneyInfoMap(api.type())
                .withFromCampaigns(new PayCampElementMap(api.type())
                        .withCampaignID(activeCampaignID1)
                        .withSum(123f))
                .withToCampaigns(new PayCampElementMap(api.type())
                        .withCampaignID(activeCampaignID2)
                        .withSum(123f)).toString();
        request = request.replaceAll("123\\.0", "[11.1]");
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(new JSONParser().parse(request),
                new AxisError(71, AxisErrorDetails.TRANSFERMONEY_SUM_SHALL_BE_FLOAT));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
    }
}
